<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-06-10
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product;

use Exception;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product;
use InvalidArgumentException;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Inventory
 *
 * Hydration class used for inserting or updating products inventories from Fastmag to Magento
 */
class Inventory extends Product
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_product_inventory';

    /**
     * @inheritDoc
     */
    public function run()
    {
        foreach ($this->jobs->getItems() as $job) {
            $fastmagRef = $job->getContentId();

            if ($this->getMagentoProductId($fastmagRef) !== false) {
                $this->currentJob = $job;
                try {
                    $hydratedData = $this->getDataFromFastmag();
                } catch (Exception $exception) {
                    $job->traceException($exception);

                    $this->logger->error(
                        __(
                            '[Job #%1] Error when updating inventory level of product with Fastmag ref "%2": %3',
                            $job->getId(),
                            $fastmagRef,
                            $exception->getMessage()
                        )->render()
                    );
                }

                $job->setHydratedData($hydratedData);

                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $exception) {
                    $job->traceException($exception);
                }
            } else {
                $this->logger->notice(
                    __(
                        'Can not change inventory level of the product with Fastmag ref "%1", as it does not exist in Magento',
                        $fastmagRef
                    )->render()
                );
            }
        }
    }

    /**
     * @inheritDoc
     */
    protected function getDataFromFastmag()
    {
        $fastmagRef = $this->currentJob->getContentId();

        if ($this->isPack($fastmagRef)) {
            throw new JobException(__(
                'Product "%1" is a pack, which can not be imported with Fastmag_Sync module',
                $fastmagRef
            ));
        }

        $hydratedData = [
            'parent_ref' => $fastmagRef,
            'is_color' => false,
            'magento_id' => $this->getMagentoProductId($fastmagRef)
        ];

        $hydratedData = array_merge($hydratedData, $this->getBasicData($fastmagRef));

        if ($this->productMustBeSynced($hydratedData)) {
            $storesList = $this->storeRepository->getList();

            foreach ($storesList as $store) {
                try {
                    $this->currentRule = $this->storesellerRuleRepository->getByStoreId($store->getId());
                } catch (Exception $exception) {
                    continue;
                }

                $fastmagStaticData = $this->getStaticData($fastmagRef);

                $hydratedDataStore['children'] = $fastmagStaticData;

                $hydratedData[$store->getCode()] = $hydratedDataStore;
            }
        }

        return $hydratedData;
    }

    /**
     * @inheritDoc
     */
    protected function getStaticData($fastmagRef)
    {
        $result = [];

        try {
            $onlyStoresellerCombination = '';
            if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_STORESELLER)) {
                $onlyStoresellerCombination = 'HAVING enable_stock = 1 ';
            }

            $stockCondition = '';
            $referenceStocks = $this->getReferenceStocks();
            if ($referenceStocks !== null && count($referenceStocks) !== 0) {
                $stockCondition = ' AND CodeMag IN ('
                    . $this->getFastmagSqlConnection()->escape($referenceStocks) . ')';
            }

            $sql = 'SELECT
                    (
                        SELECT JSON_OBJECT(
                            \'fm_product_id\', Produit,
                            \'ean13\', RIGHT(GenCod, 13),
                            \'supplier_ref\', RefFournisseur,
                            \'weight\', PoidsTC
                        )
                        FROM produits AS p
                        WHERE p.BarCode = stock.BarCode AND p.Taille = stock.Taille AND p.Couleur = stock.Couleur
                        LIMIT 1
                    ) AS combination_data,
                    (
                        SELECT 1
                        FROM stock AS s2
                        WHERE s2.BarCode = stock.BarCode AND s2.Taille = stock.Taille AND s2.Couleur = stock.Couleur
                            ' . $stockCondition . '
                        LIMIT 1
                    ) AS enable_stock,
                    stock.Taille AS size,
                    stock.Couleur AS color,
                    SUM(Stock) AS total_stock,
                    stock.AR AS in_stock
                FROM produitsfiches AS pf INNER JOIN stock ON pf.BarCode = stock.BarCode
                WHERE stock.AR = 1
                    AND stock.BarCode = ' . $this->getFastmagSqlConnection()->escape($fastmagRef, true) . '
                    ' . $stockCondition . '
                GROUP BY stock.Barcode, stock.Taille, stock.Couleur
                ' . $onlyStoresellerCombination . '
                ORDER BY stock.Couleur, stock.Taille';

            $rows = $this->getFastmagSqlConnection()->get($sql);

            if (count($rows) === 0) {
                $rows = $this->getFastmagSqlConnection()->get(str_replace('stock.AR = 1', 'stock.AR = 0', $sql));
            }
        } catch (NoConnectionException $exception) {
            throw new ProcessException($exception->getMessage());
        }

        foreach ($rows as $row) {
            $row = $this->extractJsonData($row);

            if (!array_key_exists($row['color'], $result)) {
                $result[$row['color']] = [$row['size'] => $row];
            } else {
                $result[$row['color']][$row['size']] = $row;
            }
        }

        return $result;
    }

    /**
     * @inheritDoc
     */
    protected function extractJsonData($row)
    {
        try {
            $combinationData = $this->jsonSerializer->unserialize($row['combination_data']);
            $row = array_merge($row, $combinationData);
            unset($row['combination_data']);
        } catch (InvalidArgumentException $exception) {
            $this->logger->warning(
                __(
                    '[Job #%1] Error when unserializing combination_data from Fastmag: %2',
                    $this->currentJob->getId(),
                    $row['combination_data']
                )->render()
            );
        }

        return $row;
    }
}
