<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-06-10
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product;

use Exception;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Price
 *
 * Hydration class used for inserting or updating products prices from Fastmag to Magento
 */
class Price extends Product
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_product_price';

    /**
     * @inheritDoc
     */
    public function run()
    {
        foreach ($this->jobs->getItems() as $job) {
            $fastmagRef = $job->getContentId();

            if ($this->getMagentoProductId($fastmagRef) !== false) {
                $this->currentJob = $job;

                try {
                    $hydratedData = $this->getDataFromFastmag();
                    $job->setHydratedData($hydratedData);
                } catch (JobException $exception) {
                    $job->traceException($exception);

                    $this->logger->error(
                        __(
                            '[Job #%1] Error when updating prices of product with Fastmag ref "%2": %3',
                            $job->getId(),
                            $fastmagRef,
                            $exception->getMessage()
                        )->render()
                    );
                }

                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $exception) {
                    $job->traceException($exception);
                }
            } else {
                $this->logger->notice(
                    __(
                        'Can not change prices of the product with Fastmag ref "%1", as it does not exist in Magento',
                        $fastmagRef
                    )->render()
                );
            }
        }
    }

    /**
     * @inheritDoc
     */
    protected function getDataFromFastmag()
    {
        $fastmagRef = $this->currentJob->getContentId();

        if ($this->isPack($fastmagRef)) {
            throw new JobException(__(
                'Product "%1" is a pack, which can not be imported with Fastmag_Sync module',
                $fastmagRef
            ));
        }

        $hydratedData = [
            'parent_ref' => $fastmagRef,
            'is_color'   => false,
            'magento_id' => $this->getMagentoProductId($fastmagRef)
        ];

        $hydratedData = array_merge($hydratedData, $this->getBasicData($fastmagRef));

        if ($this->productMustBeSynced($hydratedData)) {
            $storesList = $this->storeRepository->getList();

            foreach ($storesList as $store) {
                try {
                    $this->currentRule = $this->storesellerRuleRepository->getByStoreId($store->getId());
                } catch (Exception $exception) {
                    continue;
                }

                $fastmagStaticData = $this->getStaticData($fastmagRef);

                $hydratedDataStore['children'] = $fastmagStaticData;

                $hydratedData[$store->getCode()] = $hydratedDataStore;
            }
        }

        return $hydratedData;
    }

    /**
     * @inheritDoc
     */
    protected function getStaticData($fastmagRef)
    {
        $result = [];

        try {
            $rateCode = $this->currentRule->getRateCode();
            $fastmagShop = $this->currentRule->getFastmagShop();

            $standardPriceSubquery = $this->getStandardPriceSubquery($fastmagRef, $fastmagShop, $rateCode);
            $combinationPriceSubquery = $this->getCombinationPriceSubquery($fastmagRef, $rateCode);
            $priceRateSubquery = $this->getPriceRateSubquery($fastmagRef, $rateCode);
            $salesPriceSubquery = $this->getSalesPriceSubquery($fastmagRef, $rateCode);
            $salesPriceChildrenSubquery = $this->getSalesPriceChildrenSubquery($fastmagShop);
            $buyingPriceSubquery = $this->getBuyingPriceSubquery($fastmagRef);

            $onlyStoresellerCombination = '';
            if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_STORESELLER)) {
                $onlyStoresellerCombination = 'HAVING enable_stock = 1 ';
            }

            $sql = 'SELECT
                    (' . $standardPriceSubquery . ') AS standard_price,
                    (' . $combinationPriceSubquery . ') AS combination_price,
                    (' . $priceRateSubquery . ') AS price_rate,
                    (' . $salesPriceSubquery . ') AS sales_price,
                    (' . $salesPriceChildrenSubquery . ') AS sales_price_children,
                    (' . $buyingPriceSubquery . ') AS buying_price,
                    stock.Taille AS size,
                    stock.Couleur AS color
                FROM produitsfiches AS pf INNER JOIN stock ON pf.BarCode = stock.BarCode
                WHERE stock.AR = 1 AND stock.BarCode = '
                . $this->getFastmagSqlConnection()->escape($fastmagRef, true) . '
                GROUP BY stock.Barcode, stock.Taille, stock.Couleur
                ' . $onlyStoresellerCombination . '
                ORDER BY stock.Couleur, stock.Taille';

            $rows = $this->getFastmagSqlConnection()->get($sql);

            if (count($rows) === 0) {
                $rows = $this->getFastmagSqlConnection()->get(str_replace('stock.AR = 1', 'stock.AR = 0', $sql));
            }
        } catch (NoConnectionException $exception) {
            throw new ProcessException($exception->getMessage());
        }

        foreach ($rows as $row) {
            if (!array_key_exists($row['color'], $result)) {
                $result[$row['color']] = [$row['size'] => $row];
            } else {
                $result[$row['color']][$row['size']] = $row;
            }
        }

        return $result;
    }
}
