<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-07-23
 ******************************************************************************/

namespace Fastmag\Sync\Cron;

use DateInterval;
use DateTime;
use Exception;
use Fastmag\Sync\Api\LogApiRepositoryInterface as LogApiRepository;
use Fastmag\Sync\Api\LogRepositoryInterface as LogRepository;
use Fastmag\Sync\Api\Data\LogApiInterface as LogApi;
use Fastmag\Sync\Api\Data\LogInterface as Log;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\CouldNotDeleteException;

/**
 * Class LogRotate
 *
 * Cron class used for log rotation
 */
class LogRotate
{
    /** @var Config $config */
    protected $config;

    /** @var LogApiRepository $logApiRepository */
    protected $logApiRepository;

    /** @var LogRepository $logRepository */
    protected $logRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var Logger $logger */
    protected $logger;

    /**
     * Logrotation constructor
     *
     * @param Config                 $config
     * @param LogApiRepository       $logApiRepository
     * @param LogRepository          $logRepository
     * @param SearchCriteriaBuilder  $searchCriteriaBuilder
     * @param Logger                 $logger
     */
    public function __construct(
        Config $config,
        LogApiRepository $logApiRepository,
        LogRepository $logRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        Logger $logger
    ) {
        $this->config = $config;
        $this->logApiRepository = $logApiRepository;
        $this->logRepository = $logRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->logger = $logger;
    }

    /**
     * Main method
     *
     * @throws ProcessException
     */
    public function execute()
    {
        $delay = (int)$this->config->getValue(Config::XML_PATH_LOG_ROTATION_DELAY);

        if ($delay <= 0) {
            throw new ProcessException(__(
                'Logs rotation delay is not set correctly. Current value is "%". Please set a int greater than 0.',
                $delay
            ));
        }

        try {
            $limitDate = new DateTime();
            $limitDate->sub(new DateInterval('P' . $delay . 'D'));
        } catch (Exception $exception) {
            throw new ProcessException(__(
                'Logs rotation delay is not set correctly. Current value is "%". Please set a int greater than 0.',
                $delay
            ));
        }

        $this->deleteOldLogs($limitDate);
        $this->deleteOldApiLogs($limitDate);
    }

    /**
     * Delete old logs entries
     *
     * @param DateTime $limitDate
     *
     * @return void
     */
    protected function deleteOldLogs($limitDate)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(Log::CREATED_AT, $limitDate->format('Y-m-d H:i:s'), 'lt')
            ->create();

        $oldLogs = $this->logRepository->getList($searchCriteria);

        foreach ($oldLogs->getItems() as $log) {
            try {
                $this->logRepository->delete($log);
            } catch (CouldNotDeleteException $exception) {
                $this->logger->error(__('Unable to delete log #%1', $log->getId())->render());
            }
        }
    }

    /**
     * Delete old API logs entries
     *
     * @param DateTime $limitDate
     *
     * @return void
     */
    protected function deleteOldApiLogs($limitDate)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(LogApi::CREATED_AT, $limitDate->format('Y-m-d H:i:s'), 'lt')
            ->create();

        $oldApiLogs = $this->logApiRepository->getList($searchCriteria);

        foreach ($oldApiLogs->getItems() as $log) {
            try {
                $this->logApiRepository->delete($log);
            } catch (CouldNotDeleteException $exception) {
                $this->logger->error(__('Unable to delete log #%1', $log->getId())->render());
            }
        }
    }
}
