<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-07-23
 ******************************************************************************/

namespace Fastmag\Sync\Cron;

use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Sql\DirectSql;
use Laminas\Uri\Uri as UriHandler;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\HTTP\ZendClient;
use Magento\Framework\Module\ResourceInterface as ModuleResource;
use Magento\Framework\Webapi\Rest\Request;
use Magento\Store\Model\StoreManagerInterface;
use Zend_Http_Client_Exception;

/**
 * Class PushVersion
 *
 * Cron class used for log rotation
 */
class PushVersion
{
    /** @var Config $config */
    protected $config;

    /** @var ZendClient $client */
    protected $client;

    /** @var ModuleResource */
    protected $moduleResource;

    /** @var StoreManagerInterface */
    protected $storeManager;

    /** @var DirectSql $connection */
    protected $connection;

    /** @var UriHandler $uriHandler */
    protected $uriHandler;

    /**
     * PushVersion constructor.
     *
     * @param Config                $config
     * @param ZendClient            $client
     * @param ModuleResource        $moduleResource
     * @param StoreManagerInterface $storeManager
     * @param DirectSql             $connection
     * @param UriHandler            $uriHandler
     */
    public function __construct(
        Config $config,
        ZendClient $client,
        ModuleResource $moduleResource,
        StoreManagerInterface $storeManager,
        DirectSql $connection,
        UriHandler $uriHandler
    ) {
        $this->config = $config;
        $this->client = $client;
        $this->moduleResource = $moduleResource;
        $this->storeManager = $storeManager;
        $this->connection = $connection;
        $this->uriHandler = $uriHandler;
    }

    /**
     * Send module version to monitoring website
     *
     * @return void
     *
     * @throws NoSuchEntityException|NoConnectionException
     */
    public function execute()
    {
        $websiteHmioId = $this->config->getValue(Config::XML_PATH_ABOUT_WEBSITE_ID);

        $url = $this->storeManager->getStore()->getBaseUrl();
        $domain = $this->uriHandler->parse($url)->getHost();

        try {
            $this->client->setUri('https://www.home-made.io/monitoring/fastmagsync_modules_list.php')
                ->setMethod(Request::METHOD_POST)
                ->setHeaders(['Content-Type' => 'application/x-www-form-urlencoded'])
                ->setParameterPost([
                    'version'  => $this->moduleResource->getDbVersion('Fastmag_Sync'),
                    'website'  => $url,
                    'id_site'  => $websiteHmioId,
                    'path'     => $domain,
                    'customer' => 'fastmag_' . strtolower($this->config->getValue(Config::XML_PATH_CONNECT_API_CHAIN)),
                    'mysql_use_fm_triggers' => (int)$this->connection->nativeTriggersExists()
                ]);

            $this->client->request();
        } catch (Zend_Http_Client_Exception $exception) {
            throw new NoConnectionException(__('Unable to push module version: %1', $exception->getMessage()));
        }

        $response = $this->client->getLastResponse();

        if ($response && ($response->getStatus() < 200 || $response->getStatus() >= 300)) {
            throw new NoConnectionException(__(
                'Unable to push module version. Status: %1, message: %2',
                $response->getStatus(),
                $response->getMessage()
            ));
        }
    }
}
