<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-06-17
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag;

use DateInterval;
use DateTime;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Jobqueue\ToFastmagRepository;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Worker;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Clean
 *
 * Worker class used for cleaning to Fastmag jobs finished since more than 45 minutes
 */
class Clean extends Worker
{
    /** @inheritDoc */
    protected $code = 'tofastmag_clean';

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var ToFastmagRepository $jobRepository */
    protected $jobRepository;

    /**
     * Scheduled constructor.
     *
     * @param Logger                $logger
     * @param ResourceConnection    $resourceConnection
     * @param FastmagSql            $fastmagSql
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param ToFastmagRepository   $jobRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        FastmagSql $fastmagSql,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        ToFastmagRepository $jobRepository
    ) {
        parent::__construct($logger, $resourceConnection, $fastmagSql);

        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     *
     * @throws CouldNotSaveException
     * @throws CouldNotDeleteException
     */
    public function run()
    {
        $this->resetErrorJobs();
        $this->cleanDoneJobs();
    }

    /**
     * Update jobs data to allow rerun
     *
     * @throws CouldNotSaveException
     */
    protected function resetErrorJobs()
    {
        $limitDatetime = new DateTime();
        $limitDatetime->sub(new DateInterval('PT5M'));

        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(ToFastmagInterface::RETRY_COUNT, 4, 'lt')
            ->addFilter(ToFastmagInterface::STATUS, ToFastmagInterface::STATUS_ERROR)
            ->addFilter(ToFastmagInterface::PROCESSED_AT, $limitDatetime->format('Y-m-d H:i:s'), 'lt')
            ->create();

        $jobsToDelete = $this->jobRepository->getList($searchCriteria);

        foreach ($jobsToDelete->getItems() as $job) {
            $job->reset();
            $this->jobRepository->save($job);
        }
    }

    /**
     * Clean done jobs from queue
     *
     * @throws CouldNotDeleteException
     */
    protected function cleanDoneJobs()
    {
        $dateTimeLimit = new DateTime();
        $dateTimeLimit->sub(new DateInterval('PT45M'));

        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(ToFastmagInterface::STATUS, ToFastmagInterface::STATUS_OK)
            ->addFilter(ToFastmagInterface::PROCESSED_AT, $dateTimeLimit->format('Y-m-d H:i:s'), 'lt')
            ->create();

        $jobsToDelete = $this->jobRepository->getList($searchCriteria);

        foreach ($jobsToDelete->getItems() as $job) {
            $this->jobRepository->delete($job);
        }

        $this->setLastExecutionDate();
    }
}
