<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-07-23
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration\Order;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Data\OrderInterfaceFactory as SyncOrderFactory;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\OrderRepositoryInterface as SyncOrderRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Helper\Text;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Api;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\Order\Item as OrderItemEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration\Order;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * Class Save
 *
 * Integration class used for inserting or updating customers from Magento to Fastmag
 */
class Save extends Order
{
    /** @inheritDoc */
    protected $code = 'tofastmag_integration_order_save';

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /**
     * Integration constructor.
     *
     * @param Logger                $logger
     * @param ResourceConnection    $resourceConnection
     * @param FastmagSql            $fastmagSql
     * @param Config                $config
     * @param JobRepository         $jobRepository
     * @param Api                   $api
     * @param SyncOrderFactory      $syncedOrderFactory
     * @param SyncOrderRepository   $syncedOrderRepository
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param Json                  $jsonSerializer
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        FastmagSql $fastmagSql,
        Config $config,
        JobRepository $jobRepository,
        Api $api,
        SyncOrderFactory $syncedOrderFactory,
        SyncOrderRepository $syncedOrderRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        Json $jsonSerializer
    ) {
        parent::__construct(
            $logger,
            $resourceConnection,
            $fastmagSql,
            $config,
            $jobRepository,
            $api,
            $syncedOrderFactory,
            $syncedOrderRepository,
            $searchCriteriaBuilder
        );

        $this->jsonSerializer = $jsonSerializer;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->notice($exception->getMessage());

            return;
        }

        foreach ($this->jobs->getItems() as $job) {
            if (!$job->isInError()) {
                try {
                    $this->processJob($job);

                    $job->setStatus(Job::STATUS_OK)
                        ->setProcessedAt(date('Y-m-d H:i:s'));
                } catch (JobException $exception) {
                    $job->traceException($exception);

                    $this->logger->error(
                        __(
                            '[Job #%1] Error on order with Magento ID #%2: %3',
                            $job->getId(),
                            $job->getContentId(),
                            $exception->getMessage()
                        )->render()
                    );
                }

                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $exception) {
                    $job->traceException($exception);
                }
            }
        }
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        /** @var OrderEntity $orderEntity */
        $orderEntity = $job->getEntity();

        $transactionType = $orderEntity->getTransactionType();

        if ($transactionType === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_NONE) {
            throw new JobException(__('No transaction type for the current order'));
        }

        if ($orderEntity->getStoresellerRule() === null) {
            throw new JobException(__(
                'Unable to cancel order #%1 to Fastmag through EDI: no storeseller found.',
                $orderEntity->getMagentoId()
            ));
        }

        $chain = $orderEntity->getStoresellerRule()->getChain();
        $shop = $orderEntity->getStoresellerRule()->getFastmagShop();

        $ediResults = [];
        $date = date('Y-m-d H:i:s');

        switch ($transactionType) {
            case OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_CANCELLATION:
                $ediResults[$transactionType][] = $this->sendCancellation($orderEntity, $chain, $shop);
                break;
            case OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE:
                $ediResults[$transactionType][] = $this->sendReservationToSale($orderEntity, $chain, $shop);
                break;
            case OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE:
            case OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION:
                $itemsStockLevels = $this->checkForTransfers($orderEntity);

                if (count($itemsStockLevels) > 0) {
                    foreach ($itemsStockLevels as $itemId => $itemStockLevels) {
                        $ediResults[OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_TRANSFER][] =
                            $this->sendTransfers($orderEntity, $chain, $shop, $itemId, $itemStockLevels);
                    }
                }

                $ediResults[$transactionType][] = $this->sendTransaction($orderEntity, $chain, $shop);
                break;
            default:
                $ediResults[$transactionType][] = $this->sendTransaction($orderEntity, $chain, $shop);
                break;
        }

        $this->saveSyncOrder($orderEntity->getMagentoId(), $ediResults, $date, $shop);
    }

    /**
     * Send cancellation to Fastmag
     *
     * @param OrderEntity $orderEntity
     * @param string      $chain
     * @param string      $shop
     *
     * @return array
     *
     * @throws JobException
     */
    protected function sendCancellation($orderEntity, $chain, $shop)
    {
        if ($orderEntity->getLastTransaction() === null) {
            throw new JobException(__(
                'Unable to cancel order #%1 to Fastmag through EDI: no previous transaction found.',
                $orderEntity->getMagentoId()
            ));
        }

        $ediData = [
            $shop,
            $orderEntity->getLastTransaction()->getTransactionType(),
            $orderEntity->getLastTransaction()->getTransactionId(),
        ];

        $ediFormattedData = $this->edi->formatInline(Edi::EDI_HEADER_CANCELLATION_INSERT, $ediData);

        try {
            $result = $this->edi->post($ediFormattedData, ['chain' => $chain, 'shop' => $shop]);
        } catch (NoConnectionException $exception) {
            throw new JobException(__(
                'Unable to cancel order #%1 to Fastmag through EDI: %2. Data sent: %3',
                $orderEntity->getMagentoId(),
                $exception->getMessage(),
                $ediFormattedData
            ));
        }

        return $result;
    }

    /**
     * Send ReservationToSale transaction to Fastmag
     *
     * @param OrderEntity $orderEntity
     * @param string      $chain
     * @param string      $shop
     *
     * @return array
     *
     * @throws JobException
     */
    protected function sendReservationToSale($orderEntity, $chain, $shop)
    {
        $ediFormattedData = $this->getHeaderReservationToSale($orderEntity) . '|' . PHP_EOL
            . $this->getFooter($orderEntity);

        try {
            $result = $this->edi->post($ediFormattedData, ['chain' => $chain, 'shop' => $shop]);
        } catch (NoConnectionException $exception) {
            throw new JobException(__(
                'Unable to send reservation to sale transaction #%1 to Fastmag through EDI: %2. Data sent: %3',
                $orderEntity->getMagentoId(),
                $exception->getMessage(),
                $ediFormattedData
            ));
        }

        return $result;
    }

    /**
     * Check on Fastmag if the quantity ordered require stock transfers on Fastmag DB
     *
     * @param OrderEntity $orderEntity
     *
     * @return array
     *
     * @throws JobException
     */
    protected function checkForTransfers($orderEntity)
    {
        $result = [];

        if ($orderEntity->getStoresellerRule() === null) {
            throw new JobException(__(
                'Unable to check for transfer for order #%1 to Fastmag through EDI: no storeseller found.',
                $orderEntity->getMagentoId()
            ));
        }

        if ($orderEntity->getStoresellerRule()->getReferenceStock() !== null) {
            $stockCodes = explode('|', $orderEntity->getStoresellerRule()->getReferenceStock());
            array_unshift($stockCodes, $orderEntity->getStoresellerRule()->getFastmagShop());
        } else {
            $stockCodes = [$orderEntity->getStoresellerRule()->getFastmagShop()];
        }

        $items = $orderEntity->getItems();

        if (count($items) > 0) {
            foreach ($items as $itemId => $item) {
                if ($item->getFastmagBarcode() === null
                    || $item->getFastmagSize() === null
                    || $item->getFastmagColor() === null
                ) {
                    throw new JobException(__(
                        'Missing barcode/size/color for an item of the order when trying to get inventory level on Fastmag'
                    ));
                }

                $qtyOrdered = $item->getQtyOrdered();
                $barcode = $item->getFastmagBarcode();
                $size = $item->getFastmagSize();
                $color = $item->getFastmagColor();

                $inventoryLevels = $this->getFastmagInventoryLevels($barcode, $size, $color, $stockCodes);

                if ($inventoryLevels[$stockCodes[0]] < $qtyOrdered) {
                    $result[$itemId] = $inventoryLevels;
                }
            }
        }

        return $result;
    }

    /**
     * Get Fastmag inventory levels for the item
     *
     * @param string   $barcode
     * @param string   $size
     * @param string   $color
     * @param string[] $stockCodes
     *
     * @return array
     *
     * @throws JobException
     */
    protected function getFastmagInventoryLevels($barcode, $size, $color, $stockCodes)
    {
        $result = [];

        try {
            $sql = 'SELECT CodeMag AS stock_code, SUM(Stock) AS total_inventory
            FROM stock
            WHERE CodeMag IN (' . $this->getFastmagSqlConnection()->escape($stockCodes) . ')
                AND AR = 1
                AND BarCode = ' . $this->getFastmagSqlConnection()->escape($barcode) . '
                AND Taille = ' . $this->getFastmagSqlConnection()->escape($size) . '
                AND Couleur = ' . $this->getFastmagSqlConnection()->escape($color) . '
            GROUP BY CodeMag
            HAVING SUM(Stock) > 0
            LIMIT 1';

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new JobException(__(
                'Error when getting inventory level of order\'s items. Message: %2. Item barcode: %2. Item size: %3. Item color: %4. Stock code: %5',
                $exception->getMessage(),
                $barcode,
                $size,
                $color,
                implode(',', $stockCodes)
            ));
        }

        if (count($rows) > 0) {
            foreach ($rows as $row) {
                $result[$row['stock_code']] = $row['total_inventory'];
            }
        }

        return $result;
    }

    /**
     * Send transfers to Fastmag
     *
     * @param OrderEntity $orderEntity
     * @param string      $chain
     * @param string      $shop
     * @param int         $itemId
     * @param array       $itemStockLevels
     *
     * @return array
     *
     * @throws JobException
     */
    protected function sendTransfers($orderEntity, $chain, $shop, $itemId, $itemStockLevels)
    {
        $result = [];

        $itemEntity = $orderEntity->getItemByMagentoId($itemId);

        if ($itemEntity === null || count($orderEntity->getItems()) === 0) {
            throw new JobException(
                __('Item #%1 can not be transferred as it does not exist in the order in the first place', $itemId)
            );
        }

        $qtyToTransfer = $itemEntity->getQtyOrdered();
        $qtyTransferred = 0;

        foreach ($itemStockLevels as $stockCode => $inventoryLevel) {
            if ($stockCode !== $shop && $inventoryLevel < $qtyToTransfer) {
                $qtyTransferred = min($qtyToTransfer, $inventoryLevel);
                $transferData = [
                    'qty_to_transfer' => $qtyTransferred,
                    'origin_shop' => $stockCode
                ];

                $result[] = $this->sendTransfer($orderEntity, $chain, $shop, $itemId, $transferData);

                $qtyToTransfer -= $qtyTransferred;
            }

            if ($qtyToTransfer === 0) {
                $this->sendAlert(
                    $orderEntity->getMagentoId(),
                    $stockCode,
                    $shop,
                    [$itemId => $itemEntity],
                    [$itemId => $qtyTransferred]
                );

                break;
            }
        }

        return $result;
    }

    /**
     * Send transfer to Fastmag
     *
     * @param OrderEntity $orderEntity
     * @param string      $chain
     * @param string      $shop
     * @param int         $itemId
     * @param array       $transferData
     *
     * @return array
     *
     * @throws JobException
     */
    protected function sendTransfer($orderEntity, $chain, $shop, $itemId, $transferData)
    {
        $ediFormattedData = $this->getHeaderTransfer($orderEntity, $transferData) . '|' . PHP_EOL
            . $this->getProductLinesTransfer($orderEntity, $itemId, $transferData);

        try {
            $result = $this->edi->post($ediFormattedData, ['chain' => $chain, 'shop' => $shop]);
            $result['origin_shop'] = $transferData['origin_shop'];
        } catch (NoConnectionException $exception) {
            throw new JobException(__(
                'Unable to send order #%1 to Fastmag through EDI: %2. Data sent: %3',
                $orderEntity->getMagentoId(),
                $exception->getMessage(),
                $ediFormattedData
            ));
        }

        return $result;
    }

    /**
     * Send transaction to Fastmag
     *
     * @param OrderEntity $orderEntity
     * @param string      $chain
     * @param string      $shop
     *
     * @return array
     *
     * @throws JobException
     */
    protected function sendTransaction($orderEntity, $chain, $shop)
    {
        $ediFormattedData = $this->getHeaderTransaction($orderEntity) . '|' . PHP_EOL
            . $this->getProductLinesTransaction($orderEntity) . '|' . PHP_EOL
            . $this->getFooter($orderEntity);

        try {
            $result = $this->edi->post($ediFormattedData, ['chain' => $chain, 'shop' => $shop]);
        } catch (NoConnectionException $exception) {
            throw new JobException(__(
                'Unable to send order #%1 to Fastmag through EDI: %2. Data sent: %3',
                $orderEntity->getMagentoId(),
                $exception->getMessage(),
                $ediFormattedData
            ));
        }

        if ($orderEntity->getTransactionType() === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE) {
            $this->sendAlert($orderEntity->getMagentoId(), '', $shop, $orderEntity->getItems());
        }

        return $result;
    }

    /**
     * Get header for generic transaction
     *
     * @param OrderEntity $orderEntity
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getHeaderReservationToSale($orderEntity)
    {
        if ($orderEntity->getStoresellerRule() === null) {
            throw new JobException(__(
                'Unable to transform reservation to sale for order #%1 to Fastmag through EDI: no storeseller found.',
                $orderEntity->getMagentoId()
            ));
        }

        if ($orderEntity->getShippingAddress() === null) {
            throw new JobException(__(
                'Unable to transform reservation to sale for order #%1 to Fastmag through EDI: no shipping address found.',
                $orderEntity->getMagentoId()
            ));
        }

        if ($orderEntity->getLastTransaction() === null) {
            throw new JobException(__(
                'Unable to transform reservation to sale for order #%1 to Fastmag through EDI: no previous transaction found.',
                $orderEntity->getMagentoId()
            ));
        }

        $ediData = [
            $orderEntity->getStoresellerRule()->getFastmagShop(),
            $orderEntity->getGrandTotal(),
            $orderEntity->getTotalQty(),
            ($orderEntity->getTransactionType() === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RMA
                ? OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE : $orderEntity->getTransactionType()),
            ($orderEntity->getStoresellerRule() ? $orderEntity->getStoresellerRule()->getSeller() : ''),
            $orderEntity->getCustomer()->getFastmagId(),
            $orderEntity->getShippingAddress()->getAlias(),
            '', // Vide
            '', // Vide
            '', // Vide
            $orderEntity->getLastTransaction()->getTransactionId()
        ];

        return $this->edi->formatInline(Edi::EDI_HEADER_TRANSACTION_INSERT, $ediData);
    }

    /**
     * Get header for transfer
     *
     * @param OrderEntity $orderEntity
     * @param array       $transferData
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getHeaderTransfer($orderEntity, $transferData)
    {
        if ($orderEntity->getStoresellerRule() === null) {
            throw new JobException(__(
                'Unable to send transfer for order #%1 to Fastmag through EDI: no storeseller found.',
                $orderEntity->getMagentoId()
            ));
        }

        $ediData = [
            $transferData['origin_shop'],
            $orderEntity->getStoresellerRule()->getFastmagShop(),
            (int)$transferData['qty_to_transfer'],
            $orderEntity->getStoresellerRule()->getSeller(),
            '', // Réservé
            $this->config->isSetFlag(Config::XML_PATH_ORDER_WORKFLOW_INVENTORY_TRANSFER)
                ? Edi::EDI_TRUE : Edi::EDI_FALSE,
        ];

        return $this->edi->formatInline(Edi::EDI_HEADER_TRANSACTION_INSERT, $ediData);
    }

    /**
     * Get header for generic transaction
     *
     * @param OrderEntity $orderEntity
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getHeaderTransaction($orderEntity)
    {
        if ($orderEntity->getStoresellerRule() === null) {
            throw new JobException(__(
                'Unable to send transaction for order #%1 to Fastmag through EDI: no storeseller found.',
                $orderEntity->getMagentoId()
            ));
        }

        if ($orderEntity->getShippingAddress() === null) {
            throw new JobException(__(
                'Unable to send transaction for order #%1 to Fastmag through EDI: no shipping address found.',
                $orderEntity->getMagentoId()
            ));
        }

        $ediData = [
            $orderEntity->getStoresellerRule()->getFastmagShop(),
            $orderEntity->getGrandTotal(),
            (int)$orderEntity->getTotalQty(),
            ($orderEntity->getTransactionType() === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RMA
                ? OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE : $orderEntity->getTransactionType()),
            $orderEntity->getStoresellerRule()->getSeller(),
            $orderEntity->getCustomer()->getFastmagId(),
            $orderEntity->getShippingAddress()->getAlias(),
            '', // Réservé
            '', // Réservé
            '', // Réservé
            '', // Réservé
            '', // Réservé
            '', // Réservé
            '', // Réservé
            '', // Utilisateur
            '', // Statut de la pièce
            '', // Vente HT
            '', // Etat de la pièce
            $orderEntity->getIncrementId(),
            Edi::EDI_HEADER_TRANSACTION_DEFAULT_VALUE_ORIGIN,
            '', // Point Retrait
            ''  // Code du règlement
        ];

        return $this->edi->formatInline(Edi::EDI_HEADER_TRANSACTION_INSERT, $ediData);
    }

    /**
     * Get product lines for generic transaction
     *
     * @param OrderEntity $orderEntity
     * @param int         $itemId
     * @param array       $transferData
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getProductLinesTransfer($orderEntity, $itemId, $transferData)
    {
        $itemEntity = $orderEntity->getItemByMagentoId($itemId);

        if ($itemEntity === null) {
            throw new JobException(__('Item #%1 does not exist in order hydrated data', $itemId));
        }

        $ediData = [
            $itemEntity->getFastmagBarcode(),
            $itemEntity->getFastmagSize(),
            $itemEntity->getFastmagColor(),
            Text::removeAccents($itemEntity->getName()),
            number_format($itemEntity->getPriceInclTax(), 2, '.', ''),
            (int)$transferData['qty_to_transfer'],
            '', // Vide
            number_format($itemEntity->getPriceInclTax() * $transferData['qty_to_transfer'], 2, '.', '')
        ];

        $result = $this->edi->formatInline(Edi::EDI_LINE_TRANSACTION_PRODUCT_ID, $ediData) . '|' . PHP_EOL;

        return trim($result, '|' . PHP_EOL);
    }

    /**
     * Get product lines for generic transaction
     *
     * @param OrderEntity $orderEntity
     *
     * @return string
     */
    protected function getProductLinesTransaction($orderEntity)
    {
        $result = '';
        
        foreach ($orderEntity->getItems() as $itemId => $itemEntity) {
            $ediData = [
                $itemEntity->getFastmagBarcode(),
                $itemEntity->getFastmagSize(),
                $itemEntity->getFastmagColor(),
                Text::removeAccents($itemEntity->getName()),
                number_format($itemEntity->getPriceInclTax(), 2, '.', ''),
                (int)$itemEntity->getQtyOrdered(),
                number_format($itemEntity->getDiscountAmount(), 2, '.', ''),
                number_format($itemEntity->getRowTotalInclTax(), 2, '.', ''),
                '', // Motif
                '', // Commentaires
            ];

            $result .= $this->edi->formatInline(Edi::EDI_LINE_TRANSACTION_PRODUCT_ID, $ediData) . '|' . PHP_EOL;
        }

        return trim($result, '|' . PHP_EOL);
    }

    /**
     * Get product lines for generic transaction
     *
     * @param OrderEntity $orderEntity
     *
     * @return string
     */
    protected function getFooter($orderEntity)
    {
        $ediData = [
            $orderEntity->getPaymentCode(),
            $orderEntity->getGrandTotal(),
            '', // Numéro d’avoir ou d’acompte
            '', // Date Echéance
            $orderEntity->getPaymentId()
        ];

        return $this->edi->formatInline(Edi::EDI_FOOTER_TRANSACTION_INSERT, $ediData);
    }

    /**
     * Send alert to shop
     *
     * @param int               $orderId
     * @param string            $originShop
     * @param string            $targetShop
     * @param OrderItemEntity[] $items
     * @param array             $qtyTransferred
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendAlert($orderId, $originShop, $targetShop, $items, $qtyTransferred = [])
    {
        $message = [];

        foreach ($items as $itemId => $itemEntity) {
            $message[] = 'Ref : ' . $itemEntity->getFastmagBarcode()
                . ' - Taille : ' . $itemEntity->getFastmagSize()
                . ' - Couleur : ' . $itemEntity->getFastmagColor()
                . ' - Quantité : ';
            
            if (array_key_exists($itemId, $qtyTransferred)) {
                $message[] .= $qtyTransferred[$itemId];
            } else {
                $message[] .= $itemEntity->getQtyOrdered();
            }
        }

        try {
            $sql = 'INSERT INTO consignes (Expediteur, Destinataire, DateCreation, Sujet, Message, Slide)
                VALUES (
                    ' . $this->getFastmagSqlConnection()->escape($originShop) . ',
                    ' . $this->getFastmagSqlConnection()->escape($targetShop) . ',
                    now(),
                    ' . $this->getFastmagSqlConnection()->escape(__('Web order #%1', $orderId)) . ',
                    ' . $this->getFastmagSqlConnection()->escape(__('To prepare: %1', implode(' *** ', $message))) . '
                );';

            $this->getFastmagSqlConnection()->post($sql);
        } catch (NoConnectionException $exception) {
            throw new JobException(__(
                'Unable to send alert to shop %1: Message: %2. Message sent: %3',
                $targetShop,
                $exception->getMessage(),
                implode(' *** ', $message)
            ));
        }
    }

    /**
     * Save order synchronizations
     *
     * @param int    $orderId
     * @param array  $ediResults
     * @param string $requestAt
     * @param string $targetShop
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveSyncOrder($orderId, $ediResults, $requestAt, $targetShop)
    {
        foreach ($ediResults as $transactionType => $ediResultsByType) {
            foreach ($ediResultsByType as $ediResult) {
                $syncedOrder = $this->syncOrderFactory->create();

                $syncedOrder->setOrderId($orderId)
                    ->setTransactionId($ediResult['id'])
                    ->setTransactionIdBis($ediResult['id_bis'])
                    ->setTransactionType($transactionType)
                    ->setState($ediResult['status'])
                    ->setEdiCsv($ediResult['post_data'])
                    ->setEdiSerialized($this->jsonSerializer->serialize($ediResult))
                    ->setResultLog($ediResult['response'])
                    ->setRequestAt($requestAt)
                    ->setResultAt(date('Y-m-d H:i:s'))
                    ->setTargetShop($targetShop);

                if (array_key_exists('origin_shop', $ediResult)) {
                    $syncedOrder->setOriginShop($ediResult['origin_shop']);
                }

                try {
                    $this->syncOrderRepository->save($syncedOrder);
                } catch (Exception $exception) {
                    throw new JobException(__('Unable to save order #%1 synchronization on Magento DB', $orderId));
                }
            }
        }
    }
}
