<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-07-23
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product;

use Exception;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Price
 *
 * Hydration class used for inserting or updating products prices from Fastmag to Magento
 */
class Price extends Product
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_product_price';

    /**
     * @inheritDoc
     */
    public function run()
    {
        foreach ($this->jobs->getItems() as $job) {
            $fastmagRef = $job->getContentId();

            if ($this->getMagentoProductId($fastmagRef) !== false) {
                $this->currentJob = $job;
                $this->defaultRule = null;
                $this->entities[$this->currentJob->getContentId()] = $this->productEntityFactory->create();

                try {
                    $entity = $this->getDataFromFastmag();

                    $job->setHydratedData($entity->export())
                        ->setEntity($entity);
                } catch (JobException $exception) {
                    $job->traceException($exception);

                    $this->logger->error(
                        __(
                            '[Job #%1] Error when updating prices of product with Fastmag ref "%2": %3',
                            $job->getId(),
                            $fastmagRef,
                            $exception->getMessage()
                        )->render()
                    );
                }

                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $exception) {
                    $job->traceException($exception);
                }
            } else {
                $this->logger->notice(
                    __(
                        'Can not change prices of the product with Fastmag ref "%1", as it does not exist in Magento',
                        $fastmagRef
                    )->render()
                );
            }
        }
    }

    /**
     * @inheritDoc
     */
    protected function getDataFromFastmag()
    {
        $fastmagRef = $this->currentJob->getContentId();

        if ($this->isPack($fastmagRef)) {
            throw new JobException(__(
                'Product "%1" is a pack, which can not be imported with Fastmag_Sync module',
                $fastmagRef
            ));
        }

        $currentEntity = $this->entities[$this->currentJob->getContentId()];
        $currentEntity->setRef($fastmagRef)
            ->setMagentoId($this->getMagentoProductId($fastmagRef));

        $this->getParentData($fastmagRef);

        if ($this->productMustBeSynced($currentEntity)) {
            $storesList = $this->storeRepository->getList();

            foreach ($storesList as $store) {
                try {
                    $this->currentRule = $this->storesellerRuleRepository->getByStoreId($store->getId());
                } catch (Exception $exception) {
                    continue;
                }

                if ($this->defaultRule === null) {
                    $this->defaultRule = $this->currentRule;
                }

                $this->getChildrenData($fastmagRef);
            }
        }

        return $currentEntity;
    }

    /**
     * @inheritDoc
     */
    protected function getParentData($fastmagRef)
    {
        try {
            $sql = 'SELECT pf.VisibleWeb AS visible_web,
                    pf.Actif AS active,
                    COUNT(s.ID) AS stock_level,
                    f.TauxTVA AS vat_rate
                FROM produitsfiches AS pf LEFT JOIN stock AS s on pf.BarCode = s.BarCode
                    LEFT JOIN familles AS f ON pf.Famille = f.Famille
                WHERE pf.BarCode = ' . $this->getFastmagSqlConnection()->escape($fastmagRef, true) . '
                    AND s.AR = 1
                GROUP BY pf.BarCode';

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        $row = reset($rows);

        $currentEntity = $this->entities[$this->currentJob->getContentId()];
        $currentEntity->addData($row);
    }

    /**
     * @inheritDoc
     */
    protected function getChildrenData($fastmagRef)
    {
        try {
            $rateCode = $this->currentRule->getRateCode();
            $fastmagShop = $this->currentRule->getFastmagShop();

            $standardPriceSubquery = $this->getStandardPriceSubquery($fastmagRef, $fastmagShop, $rateCode);
            $combinationPriceSubquery = $this->getCombinationPriceSubquery($fastmagRef, $rateCode);
            $ratePriceSubquery = $this->getRatePriceSubquery($fastmagRef, $rateCode);
            $salesPriceSubquery = $this->getSalesPriceSubquery($fastmagRef, $rateCode);
            $salesPriceChildrenSubquery = $this->getSalesPriceChildrenSubquery($fastmagShop);
            $buyingPriceSubquery = $this->getBuyingPriceSubquery($fastmagRef);

            $onlyStoresellerCombination = '';
            if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_STORESELLER)) {
                $onlyStoresellerCombination = 'HAVING active_stock = 1 ';
            }

            $stockCondition = '';
            $referenceStocks = $this->getReferenceStocks();
            if ($referenceStocks !== null && count($referenceStocks) !== 0) {
                $stockCondition = ' AND CodeMag IN ('
                    . $this->getFastmagSqlConnection()->escape($referenceStocks) . ')';
            }
            $sql = 'SELECT
                    (
                        SELECT 1
                        FROM stock AS s2
                        WHERE s2.BarCode = stock.BarCode AND s2.Taille = stock.Taille AND s2.Couleur = stock.Couleur
                            ' . $stockCondition . '
                        LIMIT 1
                    ) AS active_stock,
                    pf.PrixVente AS indicative_price,
                    (' . $standardPriceSubquery . ') AS standard_price,
                    (' . $combinationPriceSubquery . ') AS combination_price,
                    (' . $ratePriceSubquery . ') AS rate_price,
                    (' . $salesPriceSubquery . ') AS sales_price,
                    (' . $salesPriceChildrenSubquery . ') AS sales_price_children,
                    (' . $buyingPriceSubquery . ') AS buying_price,
                    stock.Taille AS size,
                    stock.Couleur AS color
                FROM produitsfiches AS pf INNER JOIN stock ON pf.BarCode = stock.BarCode
                WHERE stock.AR = 1 AND stock.BarCode = '
                . $this->getFastmagSqlConnection()->escape($fastmagRef, true) . '
                GROUP BY stock.Barcode, stock.Taille, stock.Couleur
                ' . $onlyStoresellerCombination . '
                ORDER BY stock.Couleur, stock.Taille';

            $rows = $this->getFastmagSqlConnection()->get($sql);

            if (count($rows) === 0) {
                $rows = $this->getFastmagSqlConnection()->get(str_replace('stock.AR = 1', 'stock.AR = 0', $sql));
            }
        } catch (NoConnectionException $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        if (count($rows) > 0) {
            foreach ($rows as $row) {
                $this->saveChildrenData($row);
            }
        }
    }

    /**
     * Save static data into specific entity
     *
     * @param array $row
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveChildrenData($row)
    {
        $currentEntity = $this->entities[$this->currentJob->getContentId()];

        $variation = $this->variationEntityFactory->create();

        if ($this->currentRule === $this->defaultRule) {
            $variation->addData($row);

            $currentEntity->addVariation($variation, $row['color'], $row['size']);
        } else {
            $currentVariation = $currentEntity->getVariation($row['color'], $row['size']);

            if ($currentVariation === null) {
                throw new JobException(__(
                    'No variation already existing for ref "%1", color "%2", size "%3"',
                    $this->currentJob->getContentId(),
                    $row['color'],
                    $row['size']
                ));
            }

            foreach ($row as $code => $value) {
                if ($value !== ''
                    && $this->defaultRule !== $this->currentRule
                    && $value !== $currentVariation->getData($code)
                ) {
                    $variation->setData($code, $value);
                }
            }

            if ($variation->hasData()) {
                $currentVariation->addI18n($variation, $this->currentRule->getStoreId());
            }
        }
    }
}
