<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-07-23
 ******************************************************************************/

namespace Fastmag\Sync\Model;

use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Fastmag\Sync\Process\Manager\RemoteSync;
use Fastmag\Sync\Process\Manager\ToFastmag;
use Fastmag\Sync\Process\Manager\ToMagento;
use Magento\Framework\App\Config\ScopeConfigInterface;

/**
 * Class Cron
 *
 * Class called by the dedicated cron to enable the synchronization both ways
 */
class Cron
{
    /** @var string */
    public const XML_CONFIG_SYNC_REACTIVITY = 'fastmag_sync_jobqueue/advanced/reactivity';

    /** @var string */
    public const MODE_JOBQUEUE = 'jobqueue';

    /** @var string */
    public const MODE_REMOTESYNC = 'remotesync';

    /** @var string */
    public const SYNC_WAY_TO_FASTMAG = 'tofastmag';

    /** @var string */
    public const SYNC_WAY_TO_MAGENTO = 'tomagento';

    /** @var string */
    public const SYNC_WAY_BOTH = 'both';

    /** @var Logger $logger */
    protected $logger;

    /** @var ScopeConfigInterface $scopeConfig */
    protected $scopeConfig;

    /** @var RemoteSync $remoteSync */
    protected $remoteSync;

    /** @var ToMagento $toMagento */
    protected $toMagento;

    /** @var ToFastmag $toFastmag */
    protected $toFastmag;

    /** @var Proxy $proxy */
    protected $proxy;

    /**
     * Cron constructor
     *
     * @param Logger               $logger
     * @param ScopeConfigInterface $scopeConfig
     * @param RemoteSync           $remoteSyncManager
     * @param ToMagento            $toMagentoManager
     * @param ToFastmag            $toFastmag
     * @param Proxy                $proxy
     */
    public function __construct(
        Logger $logger,
        ScopeConfigInterface $scopeConfig,
        RemoteSync $remoteSyncManager,
        ToMagento $toMagentoManager,
        ToFastmag $toFastmag,
        Proxy $proxy
    ) {
        $this->logger = $logger;
        $this->scopeConfig = $scopeConfig;
        $this->remoteSync = $remoteSyncManager;
        $this->toMagento = $toMagentoManager;
        $this->toFastmag = $toFastmag;
        $this->proxy = $proxy;
    }

    /**
     * Main method
     *
     * @param string      $mode
     * @param string|null $syncWay
     * @param string|null $jobCode
     * @param string|null $jobId
     *
     * @return void
     *
     * @throws ProcessException
     * @throws NoConnectionException
     */
    public function run($mode, $syncWay = null, $jobCode = null, $jobId = null)
    {
        if ($this->checkParams($mode, $syncWay)) {
            $this->logger->debug('mode = ' . $mode . ' - sync way = ' . $syncWay);

            $this->sleep($mode);

            switch ($mode) {
                case self::MODE_JOBQUEUE:
                    if ($syncWay === self::SYNC_WAY_BOTH || $syncWay === self::SYNC_WAY_TO_FASTMAG) {
                        $this->toFastmag->run($jobCode, $jobId);
                    }
                    if ($syncWay === self::SYNC_WAY_BOTH || $syncWay === self::SYNC_WAY_TO_MAGENTO) {
                        $this->toMagento->run($jobCode, $jobId);
                    }
                    break;
                case self::MODE_REMOTESYNC:
                    $this->remoteSync->run();
                    break;
            }
        }
    }

    /**
     * Check params given to cron
     *
     * @param string $mode
     * @param string $syncWay
     *
     * @return bool
     *
     * @throws ProcessException
     */
    protected function checkParams($mode, $syncWay)
    {
        if (!$this->isModeValid($mode)) {
            $exception = new ProcessException(__('Job type "%1" does not exist, please check cron file usage', $mode));
            $this->logger->alert($exception->getMessage());

            throw $exception;
        }
        if (!$this->isSyncWayValid($syncWay, $mode)) {
            $exception = new ProcessException(
                __('Sync way "%1" does not exist, please check cron file usage', $syncWay)
            );
            $this->logger->alert($exception->getMessage());

            throw $exception;
        }

        return true;
    }

    /**
     * Check if cron mode parameter is valid
     *
     * @param string $mode
     *
     * @return bool
     */
    protected function isModeValid($mode)
    {
        return in_array($mode, [self::MODE_JOBQUEUE, self::MODE_REMOTESYNC], true);
    }

    /**
     * Check if cron syncway parameter is valid
     *
     * @param string $syncWay
     * @param string $mode
     *
     * @return bool
     */
    protected function isSyncWayValid($syncWay, $mode)
    {
        return ($mode === self::MODE_REMOTESYNC ||
            in_array($syncWay, [self::SYNC_WAY_TO_FASTMAG, self::SYNC_WAY_TO_MAGENTO, self::SYNC_WAY_BOTH], true));
    }

    /**
     * Put the cron to sleep for a duration depending of the job type
     *
     * @param string $jobType
     *
     * @return void
     */
    protected function sleep($jobType)
    {
        $sleepTime = 5;

        if ($jobType === self::MODE_REMOTESYNC && $this->proxy->useConnectionNativeSql() === false) {
            $sleepTime = 30;
        } elseif ($jobType === self::MODE_JOBQUEUE) {
            $sleepTime = $this->scopeConfig->getValue(self::XML_CONFIG_SYNC_REACTIVITY);
        }

        //sleep($sleepTime);
    }
}
