<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-08-26
 ******************************************************************************/

namespace Fastmag\Sync\Model\Rule;

use Exception;
use Fastmag\Sync\Api\Data\Rule\StoresellerInterface;
use Fastmag\Sync\Api\Data\Rule\StoresellerInterfaceFactory;
use Fastmag\Sync\Api\Data\Rule\StoresellerSearchResultsInterfaceFactory;
use Fastmag\Sync\Api\Rule\StoresellerRepositoryInterface;
use Fastmag\Sync\Model\ResourceModel\Rule\Storeseller as ResourceModel;
use Fastmag\Sync\Model\ResourceModel\Rule\Storeseller\CollectionFactory;
use Magento\Framework\Api\DataObjectHelper;
use Magento\Framework\Api\SearchCriteria\CollectionProcessor\FilterProcessor;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Reflection\DataObjectProcessor;

/**
 * Class StoresellerRepository
 *
 * Repository class for storeseller rules model
 */
class StoresellerRepository implements StoresellerRepositoryInterface
{
    /** @var ResourceModel $resource */
    protected $resource;

    /** @var StoresellerFactory $ruleFactory */
    protected $ruleFactory;

    /** @var CollectionFactory $collectionFactory */
    protected $collectionFactory;

    /** @var StoresellerSearchResultsInterfaceFactory $searchResultsFactory */
    protected $searchResultsFactory;

    /** @var DataObjectHelper $dataObjectHelper */
    protected $dataObjectHelper;

    /** @var DataObjectProcessor $dataObjectProcessor */
    protected $dataObjectProcessor;

    /** @var StoresellerInterfaceFactory $dataRuleFactory */
    protected $dataRuleFactory;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var CollectionProcessorInterface $collectionProcessor */
    private $collectionProcessor;

    /**
     * StoresellerRepository constructor
     *
     * @param ResourceModel                            $resource
     * @param StoresellerFactory                       $ruleFactory
     * @param StoresellerInterfaceFactory              $dataRuleFactory
     * @param CollectionFactory                        $collectionFactory
     * @param StoresellerSearchResultsInterfaceFactory $searchResultsFactory
     * @param DataObjectHelper                         $dataObjectHelper
     * @param DataObjectProcessor                      $dataObjectProcessor
     * @param SearchCriteriaBuilder                    $searchCriteriaBuilder
     * @param CollectionProcessorInterface|null        $collectionProcessor
     */
    public function __construct(
        ResourceModel $resource,
        StoresellerFactory $ruleFactory,
        StoresellerInterfaceFactory $dataRuleFactory,
        CollectionFactory $collectionFactory,
        StoresellerSearchResultsInterfaceFactory $searchResultsFactory,
        DataObjectHelper $dataObjectHelper,
        DataObjectProcessor $dataObjectProcessor,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        CollectionProcessorInterface $collectionProcessor = null
    ) {
        $this->resource = $resource;
        $this->ruleFactory = $ruleFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->dataObjectHelper = $dataObjectHelper;
        $this->dataRuleFactory = $dataRuleFactory;
        $this->dataObjectProcessor = $dataObjectProcessor;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->collectionProcessor = $collectionProcessor ?: $this->getCollectionProcessor();
    }

    /**
     * @inheritDoc
     */
    public function save(StoresellerInterface $rule)
    {
        try {
            $this->resource->save($rule);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $rule;
    }

    /**
     * @inheritDoc
     */
    public function getById($ruleId)
    {
        $rule = $this->ruleFactory->create();
        $this->resource->load($rule, $ruleId);
        if (!$rule->getId()) {
            throw new NoSuchEntityException(__('The rule with the "%1" ID doesn\'t exist.', $ruleId));
        }

        return $rule;
    }

    /**
     * @inheritDoc
     */
    public function getByStoreId($storeId)
    {
        $rule = $this->ruleFactory->create();
        $this->resource->loadByStoreId($rule, $storeId);
        if (!$rule->getId()) {
            throw new NoSuchEntityException(__('No rule linked to store #%1 exists in DB.', $storeId));
        }

        return $rule;
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $criteria)
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($criteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function getListByStoreId($storeId)
    {
        $criteria = $this->searchCriteriaBuilder->addFilter(StoresellerInterface::STORE_ID, $storeId)
            ->create();

        return $this->getList($criteria);
    }

    /**
     * @inheritDoc
     */
    public function delete(StoresellerInterface $rule)
    {
        try {
            $this->resource->delete($rule);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($ruleId)
    {
        return $this->delete($this->getById($ruleId));
    }

    /**
     * Retrieve collection processor
     *
     * @deprecated
     *
     * @return CollectionProcessorInterface
     */
    private function getCollectionProcessor()
    {
        if (!$this->collectionProcessor) {
            $this->collectionProcessor = ObjectManager::getInstance()->get(FilterProcessor::class);
        }

        return $this->collectionProcessor;
    }
}
