<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-07-23
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration\Customer\Address;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Process\Entity\ToFastmag\Address as AddressEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration\Address as AbstractAddress;
use Magento\Framework\Exception\CouldNotSaveException;
use stdClass;

/**
 * Class Save
 *
 * Integration class used for inserting or updating customers from Magento to Fastmag
 */
class Save extends AbstractAddress
{
    /** @inheritDoc */
    protected $code = 'tofastmag_integration_customer_address_save';

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->notice($exception->getMessage());

            return;
        }

        foreach ($this->jobs->getItems() as $job) {
            if (!$job->isInError()) {
                try {
                    $this->processJob($job);

                    $job->setStatus(Job::STATUS_OK)
                        ->setProcessedAt(date('Y-m-d H:i:s'));
                } catch (JobException $exception) {
                    $job->traceException($exception);

                    $this->logger->error(
                        __(
                            '[Job #%1] Error on customer\'s address with Magento ID #%2: %3',
                            $job->getId(),
                            $job->getContentId(),
                            $exception->getMessage()
                        )->render()
                    );
                }

                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $exception) {
                    $job->traceException($exception);
                }
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return true;
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        /** @var CustomerEntity $customerEntity */
        $customerEntity = $job->getEntity();

        if ($customerEntity->getShippingAddress() !== null) {
            $shippingAddress = $customerEntity->getShippingAddress();

            $this->sendFastmagAddress(
                $customerEntity->getMagentoId(),
                $customerEntity->getFastmagId(),
                $customerEntity->getEmailAddress(),
                $shippingAddress
            );
        }
    }

    /**
     * Send address to Fastmag
     *
     * @param int           $magentoId
     * @param int           $fastmagId
     * @param string        $emailAddress
     * @param AddressEntity $entity
     *
     * @throws JobException
     */
    protected function sendFastmagAddress($magentoId, $fastmagId, $emailAddress, $entity)
    {
        $addressExists = $this->checkFastmagAddress($fastmagId, $entity->getAlias());

        if (!$addressExists) {
            $this->createFastmagAddress($magentoId, $emailAddress, $entity);
        } else {
            $this->updateFastmagAddress($magentoId, $emailAddress, $entity);
        }
    }

    /**
     * Check if address exists on Fastmag
     *
     * @param int    $fastmagId
     * @param string $alias
     *
     * @return bool
     *
     * @throws JobException
     */
    protected function checkFastmagAddress($fastmagId, $alias)
    {
        try {
            $sql = 'SELECT AdrLivraison
                FROM clientadresse
		        WHERE AdrLivraison = ' . $this->getFastmagSqlConnection()->escape(utf8_decode($alias)) . '
		            AND Client = ' . $this->getFastmagSqlConnection()->escape($fastmagId);

            $rows = $this->getFastmagSqlConnection()->get($sql);

            return (bool)count($rows);
        } catch (Exception $exception) {
            throw new JobException(__(
                'Error when trying to check customer address on Fastmag. Message: %1. Customer ID: %2. Address alias: %3',
                $exception->getMessage(),
                $fastmagId,
                $alias
            ));
        }
    }

    /**
     * Create address on Fastmag
     *
     * @param int           $magentoId
     * @param string        $emailAddress
     * @param AddressEntity $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function createFastmagAddress($magentoId, $emailAddress, $entity)
    {
        $addressData = $this->setDataRequest($emailAddress, $entity);

        try {
            $response = $this->api->post('/boa/client/deliveryaddress/index.ips', $addressData);

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('message', $response) ? $response['message'] : '');

                throw new ApiException(__($message), $this->api->getLastRequest());
            }
        } catch (ApiException $e) {
            throw new JobException(__(
                'Unable to send address\' data of the customer #%1 on Fastmag through API. Code: %2. Message: %3. Data sent: %4',
                $magentoId,
                $e->getCode(),
                $e->getMessage(),
                $e->getRequest()
            ));
        }
    }

    /**
     * Update address on Fastmag
     *
     * @param int           $magentoId
     * @param int           $fastmagId
     * @param AddressEntity $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function updateFastmagAddress($magentoId, $fastmagId, $entity)
    {
        $this->createFastmagAddress($magentoId, $fastmagId, $entity);
    }

    /**
     * Set the address entity as intented for the API
     *
     * @param string        $emailAddress
     * @param AddressEntity $entity
     *
     * @return stdClass
     */
    protected function setDataRequest($emailAddress, $entity)
    {
        $result = [
            'Email'            => $emailAddress,
            'AdresseLivraison' => $entity->getAlias(),
            'Nom'              => $entity->getFirstname() . ' ' . $entity->getLastname(),
            'Adresse1'         => $entity->getStreetOne(),
            'Adresse2'         => $entity->getStreetTwo(),
            'CodePostal'       => $entity->getPostcode(),
            'Ville'            => $entity->getCity(),
            'Pays'             => $entity->getCountryId(),
            'Societe'          => $entity->getCompany(),
            'Telephone'        => $entity->getPhoneNumber(),
            'Archiver'         => $entity->isArchived()
        ];

        foreach ($result as $field => $data) {
            if ($data === null || $data === '') {
                unset($result[$field]);
            }
        }

        return (object)['Adresses' => [(object)$result]];
    }
}
