<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-08-26
 ******************************************************************************/

namespace Fastmag\Sync\Model\Rule;

use Exception;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterfaceFactory;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionSearchResultsInterfaceFactory;
use Fastmag\Sync\Api\Rule\OrdertransactionRepositoryInterface;
use Fastmag\Sync\Model\ResourceModel\Rule\Ordertransaction as ResourceModel;
use Fastmag\Sync\Model\ResourceModel\Rule\Ordertransaction\CollectionFactory;
use Magento\Framework\Api\DataObjectHelper;
use Magento\Framework\Api\SearchCriteria\CollectionProcessor\FilterProcessor;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Reflection\DataObjectProcessor;

/**
 * Class OrdertransactionRepository
 *
 * Repository class for ordertransaction rules model
 */
class OrdertransactionRepository implements OrdertransactionRepositoryInterface
{
    /** @var ResourceModel $resource */
    protected $resource;

    /** @var OrdertransactionFactory $ruleFactory */
    protected $ruleFactory;

    /** @var CollectionFactory $collectionFactory */
    protected $collectionFactory;

    /** @var OrdertransactionSearchResultsInterfaceFactory $searchResultsFactory */
    protected $searchResultsFactory;

    /** @var DataObjectHelper $dataObjectHelper */
    protected $dataObjectHelper;

    /** @var DataObjectProcessor $dataObjectProcessor */
    protected $dataObjectProcessor;

    /** @var OrdertransactionInterfaceFactory $dataRuleFactory */
    protected $dataRuleFactory;

    /** @var CollectionProcessorInterface $collectionProcessor */
    private $collectionProcessor;

    /**
     * OrdertransactionRepository constructor
     *
     * @param ResourceModel                                 $resource
     * @param OrdertransactionFactory                       $ruleFactory
     * @param OrdertransactionInterfaceFactory              $dataRuleFactory
     * @param CollectionFactory                             $collectionFactory
     * @param OrdertransactionSearchResultsInterfaceFactory $searchResultsFactory
     * @param DataObjectHelper                              $dataObjectHelper
     * @param DataObjectProcessor                           $dataObjectProcessor
     * @param CollectionProcessorInterface|null             $collectionProcessor
     */
    public function __construct(
        ResourceModel $resource,
        OrdertransactionFactory $ruleFactory,
        OrdertransactionInterfaceFactory $dataRuleFactory,
        CollectionFactory $collectionFactory,
        OrdertransactionSearchResultsInterfaceFactory $searchResultsFactory,
        DataObjectHelper $dataObjectHelper,
        DataObjectProcessor $dataObjectProcessor,
        CollectionProcessorInterface $collectionProcessor = null
    ) {
        $this->resource = $resource;
        $this->ruleFactory = $ruleFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->dataObjectHelper = $dataObjectHelper;
        $this->dataRuleFactory = $dataRuleFactory;
        $this->dataObjectProcessor = $dataObjectProcessor;
        $this->collectionProcessor = $collectionProcessor ?: $this->getCollectionProcessor();
    }

    /**
     * @inheritDoc
     */
    public function save(OrdertransactionInterface $rule)
    {
        try {
            $this->resource->save($rule);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $rule;
    }

    /**
     * @inheritDoc
     */
    public function getById($ruleId)
    {
        $rule = $this->ruleFactory->create();
        $this->resource->load($rule, $ruleId);
        if (!$rule->getId()) {
            throw new NoSuchEntityException(__('The rule with the "%1" ID doesn\'t exist.', $ruleId));
        }

        return $rule;
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $criteria)
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($criteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(OrdertransactionInterface $rule)
    {
        try {
            $this->resource->delete($rule);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($ruleId)
    {
        return $this->delete($this->getById($ruleId));
    }

    /**
     * Retrieve collection processor
     *
     * @deprecated
     *
     * @return CollectionProcessorInterface
     */
    private function getCollectionProcessor()
    {
        if (!$this->collectionProcessor) {
            $this->collectionProcessor = ObjectManager::getInstance()->get(FilterProcessor::class);
        }

        return $this->collectionProcessor;
    }
}
