<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-07-28
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration;

use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToMagento\Product as ProductEntity;
use Fastmag\Sync\Process\Worker\ToMagento\Integration;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Product
 *
 * Abstract class for Product related Integration workers
 */
abstract class Product extends Integration
{
    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->notice($exception->getMessage());
            return;
        }

        foreach ($this->jobs->getItems() as $job) {
            if (!$job->isInError()) {
                try {
                    $this->processJob($job);
                } catch (JobException $exception) {
                    $job->traceException($exception);

                    $this->logger->error(
                        __(
                            '[Job #%1] Error on product with Fastmag ID #%2: %3',
                            $job->getId(),
                            $this->getJobEntityId($job),
                            $exception->getMessage()
                        )->render()
                    );
                }

                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $exception) {
                    $job->traceException($exception);
                }
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ENABLE);
    }

    /**
     * Get job's entity ID
     *
     * @param Job $job
     *
     * @return string
     */
    protected function getJobEntityId($job)
    {
        /** @var ProductEntity $entity */
        $entity = $job->getEntity();

        return $entity->getRef();
    }

    /**
     * Check if products fill all the condition to be synced
     *
     * @param ProductEntity $entity
     *
     * @return bool
     */
    protected function productMustBeSynced($entity)
    {
        $result = ($entity->getActive() && $entity->getVisibleWeb())
            || !$this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ACTIVE_VISIBLEWEB);

        $onlyStoreseller = $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_STORESELLER);
        if ($onlyStoreseller && $entity->getStockLevel() === 0) {
            $result = false;
        }

        if ($entity->getMagentoId()) {
            $result = true;
        }

        return $result;
    }
}
