<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-08-17
 ******************************************************************************/

namespace Fastmag\Sync\Helper;

/**
 * Class Text
 *
 * Helper class to format text attributes
 */
class Text
{
    /**
     * Change encoding to utf8 if not encoded
     *
     * @param string $str
     *
     * @return boolean
     */
    public static function checkUtf8($str)
    {
        $len = strlen($str);

        for ($i = 0; $i < $len; $i++) {
            $c = ord($str[$i]);
            if ($c > 128) {
                if ($c > 247 || $c <= 191) {
                    return false;
                }

                if ($c > 239) {
                    $bytes = 4;
                } elseif ($c > 223) {
                    $bytes = 3;
                } else {
                    $bytes = 2;
                }

                if (($i + $bytes) > $len) {
                    return false;
                }

                while ($bytes > 1) {
                    $i++;
                    $b = ord($str[$i]);
                    if ($b < 128 || $b > 191) {
                        return false;
                    }
                    $bytes--;
                }
            }
        }

        return true;
    }

    /**
     * Set text to upper
     *
     * @param string $str
     *
     * @return string
     */
    public static function upper($str)
    {
        if (self::checkUtf8($str) !== true) {
            $str = utf8_encode($str);
        }

        /* @todo : mb_strtoupper ? */
        $str = strtoupper($str);

        $replacements = [
            'é' => 'É',
            'è' => 'È',
            'ê' => 'Ê',
            'à' => 'À',
            'â' => 'Â'
        ];

        return strtr($str, $replacements);
    }

    /**
     * Change fastmag caps accents to lower case
     *
     * @param string $str
     *
     * @return string
     */
    public static function lower($str)
    {
        if (self::checkUtf8($str) !== true) {
            $str = utf8_encode($str);
        }

        $str = mb_strtolower($str);

        $replacements = [
            'É' => 'é',
            'È' => 'è',
            'Ê' => 'ê',
            'À' => 'à',
            'Â' => 'â'
        ];

        return strtr($str, $replacements);
    }

    /**
     * Capitalize names
     *
     * @param string $str
     *
     * @return string
     */
    public static function capitalize($str)
    {
        $str = ucwords(self::lower($str), " \t\r\n\f\v'-");

        return $str;
    }

    /**
     * Clean string from accents
     *
     * @param string $str
     *
     * @return string
     */
    public static function removeAccents($str)
    {
        if (self::checkUtf8($str) !== true) {
            $str = utf8_encode($str);
        }

        $toReplace = [
            'À', 'Á', 'Â', 'Ã', 'Ä', 'Å', 'à', 'á', 'â', 'ã', 'ä', 'å',
            'Ç', 'ç',
            'È', 'É', 'Ê', 'Ë', 'è', 'é', 'ê', 'ë',
            'Ì', 'Í', 'Î', 'Ï', 'ì', 'í', 'î', 'ï',
            'Ñ', 'ñ',
            'Ò', 'Ó', 'Ô', 'Õ', 'Ö', 'Ø', 'ò', 'ó', 'ô', 'õ', 'ö', 'ø',
            'ß',
            'Ù', 'Ú', 'Û', 'Ü', 'ù', 'ú', 'û', 'ü',
            'ÿ'
        ];
        $toReplaceBy = [
            'A', 'A', 'A', 'A', 'A', 'A', 'a', 'a', 'a', 'a', 'a', 'a',
            'C', 'c',
            'E', 'E', 'E', 'E', 'e', 'e', 'e', 'e',
            'I', 'I', 'I', 'I', 'i', 'i', 'i', 'i',
            'N', 'n',
            'O', 'O', 'O', 'O', 'O', 'O', 'o', 'o', 'o', 'o', 'o', 'o',
            'ss',
            'U', 'U', 'U', 'U', 'u', 'u', 'u', 'u',
            'y'
        ];

        return str_replace($toReplace, $toReplaceBy, $str);
    }

    /**
     * Sanitize string
     *
     * @param string $str
     *
     * @return string
     */
    public static function sanitize($str)
    {
        $a_replace = [
            '’' => '',
            '´' => '',
            '(' => '',
            ')' => '',
            '|' => '',
            'º' => '',
            '®' => '',
            '¿' => ' ',
        ];

        return strtr($str, $a_replace);
    }
}
