<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-07-27
 ******************************************************************************/

namespace Fastmag\Sync\Setup\Patch\Data;

use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Magento\Catalog\Model\Product;
use Magento\Catalog\Model\Product\Type;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Eav\Model\Entity\Attribute\ScopedAttributeInterface;
use Magento\Eav\Setup\EavSetup;
use Magento\Eav\Setup\EavSetupFactory;
use Magento\Framework\App\Config\Storage\WriterInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Setup\ModuleDataSetupInterface;
use Magento\Framework\Setup\Patch\DataPatchInterface;
use Magento\Framework\Setup\Patch\PatchRevertableInterface;
use Zend_Validate_Exception;

/**
 * Class InstallProductAttributes
 *
 * Installer for product attributes
 */
class InstallProductAttributes implements DataPatchInterface, PatchRevertableInterface
{
    /** @var ModuleDataSetupInterface */
    protected $moduleDataSetup;

    /** @var EavSetupFactory $eavSetupFactory */
    protected $eavSetupFactory;

    /** @var EavSetup $eavSetup */
    protected $eavSetup;

    /** @var WriterInterface $configWriter */
    protected $configWriter;

    /** @var Config $config */
    protected $config;

    /** @var array $defaultValues */
    protected $defaultValues = [
        'type'         => 'int',
        'input'        => 'select',
        'user_defined' => true,
        'required'     => false,
        'group'        => Constants::ATTRIBUTE_GROUP_PRODUCT_FASTMAG_CODE,
        'apply_to'     => Type::TYPE_SIMPLE . ',' . Configurable::TYPE_CODE,
        'global'       => ScopedAttributeInterface::SCOPE_GLOBAL
    ];

    /**
     * InstallProductAttributes constructor
     *
     * @param ModuleDataSetupInterface $moduleDataSetup
     * @param EavSetupFactory          $eavSetupFactory
     * @param WriterInterface          $configWriter
     * @param Config                   $config
     */
    public function __construct(
        ModuleDataSetupInterface $moduleDataSetup,
        EavSetupFactory $eavSetupFactory,
        WriterInterface $configWriter,
        Config $config
    ) {
        $this->moduleDataSetup = $moduleDataSetup;
        $this->eavSetupFactory = $eavSetupFactory;
        $this->configWriter = $configWriter;
        $this->config = $config;
    }

    /**
     * @inheritdoc
     */
    public static function getDependencies()
    {
        return [];
    }

    /**
     * @inheritdoc
     */
    public function getAliases()
    {
        return [];
    }

    /**
     * @inheritDoc
     *
     * @throws LocalizedException
     * @throws Zend_Validate_Exception
     */
    public function apply()
    {
        $this->eavSetup = $this->eavSetupFactory->create(['setup' => $this->moduleDataSetup]);

        foreach ($this->getProductAttributesData() as $code => $data) {
            $this->eavSetup->addAttribute(
                Product::ENTITY,
                $code,
                array_merge($this->defaultValues, $data)
            );
        }

        $this->setAttributesSizeColorValue();
    }

    /**
     * @inheritDoc
     */
    public function revert()
    {
        $eavSetup = $this->eavSetupFactory->create(['setup' => $this->moduleDataSetup]);

        $attributes = array_keys($this->getProductAttributesData());
        foreach ($attributes as $code) {
            $eavSetup->removeAttribute(Product::ENTITY, $code);
        }
    }

    /**
     * Get all Fastmag related attributes data
     *
     * @return array[]
     */
    protected function getProductAttributesData()
    {
        return [
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_EAN13_CODE => [
                'label'      => 'Fastmag EAN13',
                'type'       => 'varchar',
                'input'      => 'text',
                'group'      => Constants::ATTRIBUTE_GROUP_PRODUCT_DEFAULT_CODE,
                'apply_to'   => Type::TYPE_SIMPLE,
                'sort_order' => 501
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_SIZE_CODE => [
                'label'      => 'Fastmag Size',
                'group'      => Constants::ATTRIBUTE_GROUP_PRODUCT_DEFAULT_CODE,
                'apply_to'   => Type::TYPE_SIMPLE . ',' . Type::TYPE_VIRTUAL . ',' . Configurable::TYPE_CODE,
                'sort_order' => 502,
                'searchable' => true,
                'filterable' => true,
                'visible_in_advanced_search' => true
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_COLOR_CODE => [
                'label'      => 'Fastmag Color',
                'group'      => Constants::ATTRIBUTE_GROUP_PRODUCT_DEFAULT_CODE,
                'apply_to'   => Type::TYPE_SIMPLE . ',' . Type::TYPE_VIRTUAL . ',' . Configurable::TYPE_CODE,
                'sort_order' => 503,
                'searchable' => true,
                'filterable' => true,
                'visible_in_advanced_search' => true
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_INTERNAL_COLOR_CODE => [
                'label'      => 'Fastmag Internal Color',
                'group'      => Constants::ATTRIBUTE_GROUP_PRODUCT_DEFAULT_CODE,
                'apply_to'   => Type::TYPE_SIMPLE,
                'sort_order' => 504
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_BRAND_CODE => [
                'label'      => 'Fastmag Brand',
                'sort_order' => 10
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_SUPPLIER_CODE => [
                'label'      => 'Fastmag Supplier',
                'sort_order' => 20
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_FAMILY_CODE => [
                'label'      => 'Fastmag Family',
                'sort_order' => 30
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_SUBFAMILY_CODE => [
                'label'      => 'Fastmag Subfamily',
                'sort_order' => 40
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_SECTION_CODE => [
                'label'      => 'Fastmag Section',
                'sort_order' => 50
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_SEASON_CODE => [
                'label'      => 'Fastmag Season',
                'sort_order' => 60
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_MATERIAL_CODE => [
                'label'      => 'Fastmag Material',
                'sort_order' => 70
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_MODEL_CODE => [
                'label'      => 'Fastmag Model',
                'sort_order' => 80
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_THEME_CODE => [
                'label'      => 'Fastmag Theme',
                'sort_order' => 90
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_WASHING_CODE => [
                'label'      => 'Washing Symbol',
                'sort_order' => 100
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_CHLORINE_CODE => [
                'label'      => 'Chlorine Symbol',
                'sort_order' => 110
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_IRONING_CODE => [
                'label'      => 'Ironing Symbol',
                'sort_order' => 120
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_DRY_CLEANING_CODE => [
                'label'      => 'Dry Cleaning Symbol',
                'sort_order' => 130
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_DRYING_CODE => [
                'label'      => 'Drying Symbol',
                'sort_order' => 140
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_WATER_CODE => [
                'label'      => 'Water Symbol',
                'sort_order' => 150
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_DISCOUNT_RATE_CODE => [
                'type'       => 'varchar',
                'label'      => 'Fastmag Discount Rate',
                'input'      => 'text',
                'global'     => ScopedAttributeInterface::SCOPE_WEBSITE,
                'sort_order' => 160
            ],
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_INVENTORY_SHOP_CODE => [
                'type'       => 'text',
                'label'      => 'Stock Per Shop (Json)',
                'input'      => 'textarea',
                'global'     => ScopedAttributeInterface::SCOPE_WEBSITE,
                'apply_to'   => Type::TYPE_SIMPLE,
                'sort_order' => 170
            ],
        ];
    }

    /**
     * Set color and size attributes IDs as value on related config field
     *
     * return void
     */
    protected function setAttributesSizeColorValue()
    {
        $this->setAttributeIdAsConfigValue(
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_COLOR_CODE,
            Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_COLOR
        );
        $this->setAttributeIdAsConfigValue(
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_INTERNAL_COLOR_CODE,
            Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_INTERNAL_COLOR
        );
        $this->setAttributeIdAsConfigValue(
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_SIZE_CODE,
            Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_SIZE
        );

        $this->config->clean();
    }

    /**
     * Generic method setting an attribute ID as config field value
     *
     * @param string $attributeCode
     * @param string $configPath
     *
     * @return void
     */
    protected function setAttributeIdAsConfigValue($attributeCode, $configPath)
    {
        $attributeId = $this->eavSetup->getAttributeId(Product::ENTITY, $attributeCode);

        if ($attributeId !== false) {
            $this->configWriter->save($configPath, $attributeId);
        }
    }
}
