<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-07-28
 ******************************************************************************/

namespace Fastmag\Sync\Process\Entity\ToMagento;

use DateTime;
use Exception;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToMagento\Product\Variation;
use Fastmag\Sync\Process\EntityInterface;
use IntlDateFormatter;
use Magento\Framework\DataObject;

/**
 * Class Product
 *
 * Entity class for product.
 * Save and transform data from Fastmag for Magento easy integration,
 * and transfer it from hydration worker to integration worker
 *
 * @method array|null getColorsI18n()
 * @method array|null getSizesI18n()
 * @method string|null getRef()
 * @method int|null getMagentoId()
 * @method string|null getStocksList()
 * @method float|null getVatRate()
 * @method string|null getBrand()
 * @method string|null getSupplier()
 * @method string|null getFamily()
 * @method string|null getSubfamily()
 * @method string|null getSection()
 * @method string|null getSeason()
 * @method string|null getModel()
 * @method string|null getMaterial()
 * @method string|null getTheme()
 * @method float|null getWeight()
 * @method string|null getDesignation()
 * @method string|null getDesignationBis()
 * @method string|null getDescription()
 * @method string|null getWashing()
 * @method string|null getChlorine()
 * @method string|null getIroning()
 * @method string|null getDryCleaning()
 * @method string|null getDrying()
 * @method string|null getWater()
 * @method array|null getVariations()
 * @method Product setRef(string $ref)
 * @method Product setColorsI18n(array $colorsI18n)
 * @method Product setSizesI18n(array $sizesI18n)
 * @method Product setMagentoId(int $magentoId)
 */
class Product extends DataObject implements EntityInterface
{
    /** @var string */
    public const EMPTY_COLOR_SIZE = '__empty__';

    /** @var Config $scopeConfig */
    protected $config;

    /** @var IntlDateFormatter $dateFormatter */
    protected $dateFormatter;

    /**
     * Product constructor.
     *
     * @param Config $config
     * @param array  $data
     */
    public function __construct(Config $config, array $data = [])
    {
        parent::__construct($data);

        $this->config = $config;
        $this->dateFormatter = new IntlDateFormatter(
            'fr_FR',
            IntlDateFormatter::SHORT,
            IntlDateFormatter::NONE,
            'Europe/Paris',
            IntlDateFormatter::GREGORIAN,
            'dd/MM/yyyy'
        );
    }

    /**
     * Get visible web
     *
     * @return bool|null
     */
    public function getVisibleWeb()
    {
        return $this->getData('visible_web') !== null ? (bool)$this->getData('visible_web') : null;
    }

    /**
     * Get active
     *
     * @return bool|null
     */
    public function getActive()
    {
        return $this->getData('active') !== null ? (bool)$this->getData('active') : null;
    }

    /**
     * Get creation date
     *
     * @return string|null
     */
    public function getCreationDate()
    {
        $result = null;

        $creationDate = $this->getData('creation_date');

        if (!empty($creationDate) && !($creationDate instanceof DateTime)) {
            try {
                $this->setData('creation_date', new DateTime($creationDate));

                $result = $this->dateFormatter->format($this->getData('creation_date'));
            } catch (Exception $exception) {
                // Do nothing
            }
        }

        return $result;
    }

    /**
     * Get stock level
     *
     * @return int|null
     */
    public function getStockLevel()
    {
        return $this->getData('stock_level') !== null ? (int)$this->getData('stock_level') : null;
    }

    /**
     * Get i18ns
     *
     * @return Product[]
     */
    public function getI18ns()
    {
        return $this->getData('i18ns') !== [] ? $this->getData('i18ns') : null;
    }

    /**
     * Get i18n
     *
     * @param int $storeId
     *
     * @return Product|null
     */
    public function getI18n($storeId)
    {
        $i18ns = $this->getData('i18ns');

        return $i18ns[$storeId] ?? null;
    }

    /**
     * Returns all i18ns for a specific code
     *
     * @param string $code
     *
     * @return array
     */
    public function getDataI18n($code)
    {
        $result = [];

        $i18ns = $this->getData('i18ns');

        if (is_array($i18ns) && count($i18ns) > 0) {
            /** @var Product $i18n */
            foreach ($i18ns as $storeId => $i18n) {
                if ($i18n->getData($code) !== null) {
                    $result[$storeId] = $i18n->getData($code);
                }
            }
        }

        return $result;
    }

    /**
     * Get variation
     *
     * @param string $color
     * @param string $size
     *
     * @return Variation
     */
    public function getVariation($color, $size)
    {
        $result = null;

        $variations = $this->getData('variations');

        if (is_array($variations)
            && array_key_exists($color, $variations)
            && array_key_exists($size, $variations[$color])
        ) {
            $result = $variations[$color][$size];
        }

        return $result;
    }

    /**
     * Get variations as a flat array
     *
     * @return Variation[]
     */
    public function getVariationsFlat()
    {
        $result = [];

        $variations = $this->getVariations();

        if (is_array($variations) && count($variations) > 0) {
            foreach ($variations as $sizeVariations) {
                foreach ($sizeVariations as $variation) {
                    $result[] = $variation;
                }
            }
        }

        return $result;
    }

    /**
     * Set VAT rate
     *
     * @param string $vatRate
     *
     * @return Product
     */
    public function setVatRate($vatRate)
    {
        return $this->setData('vat_rate', (float)$vatRate);
    }

    /**
     * Add i18n
     *
     * @param Product $i18n
     * @param string  $storeId
     *
     * @return Product
     */
    public function addI18n($i18n, $storeId)
    {
        $i18ns = $this->getData('i18ns');

        if ($i18ns === null) {
            $i18ns = [];
        }

        $i18ns[(int)$storeId] = $i18n;

        return $this->setData('i18ns', $i18ns);
    }

    /**
     * Add variation
     *
     * @param Variation $variation
     * @param string    $color
     * @param string    $size
     *
     * @return Product
     */
    public function addVariation($variation, $color, $size)
    {
        if (empty($color)) {
            $color = self::EMPTY_COLOR_SIZE;
        }
        if (empty($size)) {
            $size = self::EMPTY_COLOR_SIZE;
        }

        $variations = $this->getData('variations');

        if ($variations === null || !array_key_exists($color, $variations)) {
            $variations[$color] = [$size => $variation];
        } else {
            $variations[$color][$size] = $variation;
        }

        return $this->setData('variations', $variations);
    }

    /**
     * @inheritDoc
     */
    public function export()
    {
        $result = [
            'colors_i18n'     => $this->getColorsI18n(),
            'sizes_i18n'      => $this->getSizesI18n(),
            'ref'             => $this->getRef(),
            'magento_id'      => $this->getMagentoId(),
            'visible_web'     => $this->getVisibleWeb(),
            'active'          => $this->getActive(),
            'creation_date'   => $this->getCreationDate(),
            'vat_rate'        => $this->getVatRate(),
            'stock_level'     => $this->getStockLevel(),
            'stocks_list'     => $this->getStocksList(),
            'brand'           => $this->getBrand(),
            'supplier'        => $this->getSupplier(),
            'family'          => $this->getFamily(),
            'subfamily'       => $this->getSubfamily(),
            'section'         => $this->getSection(),
            'season'          => $this->getSeason(),
            'model'           => $this->getModel(),
            'material'        => $this->getMaterial(),
            'theme'           => $this->getTheme(),
            'weight'          => $this->getWeight(),
            'designation'     => $this->getDesignation(),
            'designation_bis' => $this->getDesignationBis(),
            'description'     => $this->getDescription(),
            'washing'         => $this->getWashing(),
            'chlorine'        => $this->getChlorine(),
            'ironing'         => $this->getIroning(),
            'dry_cleaning'    => $this->getDryCleaning(),
            'drying'          => $this->getDrying(),
            'water'           => $this->getWater(),
            'i18ns'           => [],
            'variations'      => []
        ];

        $i18ns = $this->getData('i18ns');
        if (is_array($i18ns) && count($i18ns) > 0) {
            foreach ($i18ns as $storeId => $i18n) {
                $result['i18ns'][$storeId] = $i18n->export();
            }
        }

        $variations = $this->getVariations();
        if (is_array($variations) && count($variations) > 0) {
            foreach ($variations as $color => $variationsSizes) {
                if (!array_key_exists($color, $result['variations'])) {
                    $result['variations'][$color] = [];
                }

                /** @var Variation $variation */
                foreach ($variationsSizes as $size => $variation) {
                    $result['variations'][$color][$size] = $variation->export();
                }
            }
        }

        return array_filter(
            $result,
            static function ($value) {
                return $value !== null && $value !== [];
            }
        );
    }
}
