<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-07-23
 ******************************************************************************/

namespace Fastmag\Sync\Console;

use Exception;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Fastmag\Sync\Process\Manager\ToFastmag;
use Fastmag\Sync\Process\Manager\ToMagento;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * Class ProcessJobsQueueCommand
 *
 * Command class for jobs queue process
 */
class ProcessJobsQueueCommand extends Command
{
    /** @var string */
    public const INPUT_KEY_SYNC_WAY = 'sync-way';

    /** @var string */
    public const INPUT_KEY_JOB_CODE = 'job-code';

    /** @var string */
    public const INPUT_KEY_JOB_ID = 'job-id';

    /** @var string */
    public const SYNC_WAY_TO_FASTMAG = 'tofastmag';

    /** @var string */
    public const SYNC_WAY_TO_MAGENTO = 'tomagento';

    /** @var string */
    public const SYNC_WAY_BOTH = 'both';

    /** @var Logger $logger */
    protected $logger;

    /** @var Config $config */
    protected $config;

    /** @var ToMagento $toMagento */
    protected $toMagento;

    /** @var ToFastmag $toFastmag */
    protected $toFastmag;

    /** @var Proxy $proxy */
    protected $proxy;

    /**
     * ProcessJobsQueueCommand constructor
     *
     * @param Logger    $logger
     * @param Config    $config
     * @param ToMagento $toMagento
     * @param ToFastmag $toFastmag
     * @param Proxy     $proxy
     */
    public function __construct(
        Logger $logger,
        Config $config,
        ToMagento $toMagento,
        ToFastmag $toFastmag,
        Proxy $proxy
    ) {
        parent::__construct();

        $this->logger = $logger;
        $this->config = $config;
        $this->toMagento = $toMagento;
        $this->toFastmag = $toFastmag;
        $this->proxy = $proxy;
    }

    /**
     * @inheritDoc
     *
     * @return void
     */
    protected function configure()
    {
        $this->setName('fastmag-sync:jobqueue');
        $this->setDescription('Process local jobs queues');

        $values = [self::SYNC_WAY_TO_FASTMAG, self::SYNC_WAY_TO_MAGENTO, self::SYNC_WAY_BOTH];

        $this->setDefinition(
            [
                new InputArgument(
                    self::INPUT_KEY_SYNC_WAY,
                    InputArgument::OPTIONAL,
                    'Sync way ("' . implode('", "', $values) . '")',
                    self::SYNC_WAY_BOTH
                ),
                new InputOption(
                    self::INPUT_KEY_JOB_CODE,
                    '-c',
                    InputOption::VALUE_OPTIONAL,
                    'Job code'
                ),
                new InputOption(
                    self::INPUT_KEY_JOB_ID,
                    '-i',
                    InputOption::VALUE_OPTIONAL,
                    'Job ID (in Magento DB)'
                )
            ]
        );

        parent::configure();
    }

    /**
     * @inheritDoc
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $syncWay = $input->getArgument(self::INPUT_KEY_SYNC_WAY);
        $jobCode = $input->getOption(self::INPUT_KEY_JOB_CODE);
        $jobId = $input->getOption(self::INPUT_KEY_JOB_ID);

        if ($this->isSyncWayValid($syncWay)) {
            $this->logger->debug('Beginning jobs queues processing');

            sleep($this->config->getValue(Config::XML_PATH_JOBQUEUE_ADVANCED_REACTIVITY));

            try {
                if ($syncWay === self::SYNC_WAY_BOTH || $syncWay === self::SYNC_WAY_TO_FASTMAG) {
                    $this->toFastmag->run($jobCode, $jobId);
                }
                if ($syncWay === self::SYNC_WAY_BOTH || $syncWay === self::SYNC_WAY_TO_MAGENTO) {
                    $this->toMagento->run($jobCode, $jobId);
                }
            } catch (Exception $exception) {
                $output->writeln($exception->getMessage() . "\n" . $exception->getTraceAsString());
            }
        }
    }

    /**
     * Check if sync way parameter is valid
     *
     * @param string $syncWay
     *
     * @return bool
     */
    protected function isSyncWayValid($syncWay)
    {
        return in_array($syncWay, [self::SYNC_WAY_TO_FASTMAG, self::SYNC_WAY_TO_MAGENTO, self::SYNC_WAY_BOTH], true);
    }
}
