<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-07-23
 ******************************************************************************/

namespace Fastmag\Sync\Process;

use DateTime;
use DateTimeZone;
use Exception;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\Collection;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Model\System\Connection\Sql\SqlInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\Pdo\Mysql;

/**
 * Class Worker
 *
 * Class handling basic operation on the workers, and inherited to hydration and integration workers
 */
abstract class Worker
{
    /** @var string */
    public const TIMEZONE_FASTMAG = 'Europe/Paris';

    /** @var string */
    public const TIMEZONE_UTC = 'UTC';

    /** @var string $code */
    protected $code;

    /** @var Logger $logger */
    protected $logger;

    /** @var ResourceConnection $resourceConnection */
    protected $resourceConnection;

    /** @var FastmagSql $fastmagSql */
    protected $fastmagSql;

    /** @var string[] $subordinateWorkersBefore */
    protected $subordinateWorkersBefore = [];

    /** @var string[] $subordinateWorkersAfter */
    protected $subordinateWorkersAfter = [];

    /** @var Collection $jobs */
    protected $jobs;

    /** @var SqlInterface $connection */
    protected $connection = false;

    /**
     * Worker constructor.
     *
     * @param Logger             $logger
     * @param ResourceConnection $resourceConnection
     * @param FastmagSql         $fastmagSql
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        FastmagSql $fastmagSql
    ) {
        $this->logger = $logger;
        $this->resourceConnection = $resourceConnection;
        $this->fastmagSql = $fastmagSql;
    }

    /**
     * @return string
     */
    public function getCode()
    {
        return $this->code;
    }

    /**
     * Main method
     *
     * @return void
     */
    abstract public function run();

    /**
     * Get manager last execution date
     *
     * Returns date as string if there is an Exception threw by DateTime constructor
     *
     * @return DateTime|string
     */
    public function getLastExecutionDate()
    {
        $select = $this->resourceConnection->getConnection()
            ->select()
            ->from('fastmag_sync_job_last_execution', 'last_executed_at')
            ->where('job_code = \'' . $this->getCode() . '\'');

        $value = $this->resourceConnection->getConnection()->fetchOne($select);

        try {
            if ($value === false) {
                $result = new DateTime('1970-01-01');
            } else {
                $result = new DateTime($value);
            }
        } catch (Exception $exception) {
            $result = $value;
        }

        return $result;
    }

    /**
     * Set manager last execution date
     *
     * @param Datetime|string $date
     *
     * @return bool
     */
    public function setLastExecutionDate($date = null)
    {
        if ($date === null) {
            try {
                $date = new DateTime();
            } catch (Exception $exception) {
                $date = date(Mysql::DATETIME_FORMAT);
            }
        }

        if (get_class($date) === 'DateTime') {
            $date = $date->format(Mysql::DATETIME_FORMAT);
        }

        $data = [
            'job_code'         => $this->getCode(),
            'last_executed_at' => $date
        ];

        $affectedRows = $this->resourceConnection->getConnection()
            ->insertOnDuplicate(
                'fastmag_sync_job_last_execution',
                $data,
                ['last_executed_at']
            );

        return ($affectedRows === 1);
    }

    /**
     * Tells if the current worker has some subordinate workers to be run before it
     *
     * @return bool
     */
    public function hasSubordinateWorkersBefore()
    {
        return count($this->subordinateWorkersBefore) > 0;
    }

    /**
     * Returns the list of workers codes to be run before the current worker
     *
     * @return string[]
     */
    public function getSubordinateWorkersBefore()
    {
        return $this->subordinateWorkersBefore;
    }

    /**
     * Tells if the current worker has some subordinate workers to be run after it
     *
     * @return bool
     */
    public function hasSubordinateWorkersAfter()
    {
        return count($this->subordinateWorkersAfter) > 0;
    }

    /**
     * Returns the list of workers codes to be run after the current worker
     *
     * @return string[]
     */
    public function getSubordinateWorkersAfter()
    {
        return $this->subordinateWorkersAfter;
    }

    /**
     * @return Collection
     */
    public function getJobs()
    {
        return $this->jobs;
    }

    /**
     * @param Collection $jobs
     *
     * @return self
     */
    public function setJobs($jobs)
    {
        $this->jobs = $jobs;

        return $this;
    }

    /**
     * Returns string if there is an exception threw by DateTime constructor
     *
     * @param string $date
     *
     * @return DateTime|string
     */
    protected function getUtcDateTimeFromFastmag($date)
    {
        try {
            $dateTime = new DateTime($date, new DateTimeZone(self::TIMEZONE_FASTMAG));
            $dateTime->setTimezone(new DateTimeZone(self::TIMEZONE_UTC));
        } catch (Exception $exception) {
            $dateTime = $date;
        }

        return $dateTime;
    }

    /**
     * Check if the SQL connection to Fastmag is working
     *
     * @return SqlInterface|false
     *
     * @throws NoConnectionException
     */
    protected function getFastmagSqlConnection()
    {
        if ($this->connection === false) {
            $this->connection = $this->fastmagSql->getConnection();
        }

        return $this->connection;
    }
}
