<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-09
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;

use Exception;
use Fastmag\Sync\Api\Rule\StoresellerRepositoryInterface as StoresellerRepository;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Model\Importer;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Entity\ToMagento\Product as ProductEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Product\Variation as VariationEntity;
use Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;
use Magento\Catalog\Api\CategoryListInterface as CategoryList;
use Magento\Catalog\Api\CategoryRepositoryInterface as CategoryRepository;
use Magento\Catalog\Api\Data\CategoryInterfaceFactory as CategoryFactory;
use Magento\Catalog\Api\Data\ProductAttributeInterface as ProductAttribute;
use Magento\Catalog\Api\ProductAttributeRepositoryInterface as AttributeRepository;
use Magento\Catalog\Api\ProductRepositoryInterface as ProductRepository;
use Magento\Catalog\Api\Data\CategoryInterface as CategoryModel;
use Magento\Catalog\Helper\Data as CatalogHelper;
use Magento\Catalog\Model\Product\Type;
use Magento\Catalog\Model\ResourceModel\Category\Tree;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Eav\Api\AttributeSetRepositoryInterface as AttributeSetRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\ImportExport\Model\Import;
use Magento\ImportExport\Model\Import\ErrorProcessing\ProcessingErrorAggregatorInterface;
use Magento\Store\Api\StoreRepositoryInterface as StoreRepository;
use Magento\Store\Api\WebsiteRepositoryInterface as WebsiteRepository;
use Magento\Tax\Api\TaxClassRepositoryInterface as TaxClassRepository;
use Magento\Tax\Model\Config as TaxConfig;

/**
 * Class Save
 *
 * Integration worker for tomagento_integration_product_create and tomagento_integration_product_update jobs
 */
class Save extends Product
{
    /** @var string */
    public const PROCESS_CATEGORY_NAME = 'Produits à traiter';

    /** @inheritDoc */
    protected $code = 'tomagento_integration_product_save';

    /** @var CategoryRepository $categoryRepository */
    protected $categoryRepository;

    /** @var CategoryFactory $categoryFactory */
    protected $categoryFactory;

    /** @var CategoryList $categoryList */
    protected $categoryList;

    /** @var Importer $importer */
    protected $importer;

    /** @var AttributeSetRepository $attributeSetRepository */
    protected $attributeSetRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var Tree $tree */
    protected $tree;

    /** @var StoresellerRepository $storesellerRuleRepository */
    protected $storesellerRuleRepository;

    /** @var WebsiteRepository $websiteRepository */
    protected $websiteRepository;

    /** @var StoreRepository $storeRepository */
    protected $storeRepository;

    /** @var TaxClassRepository $taxClassRepository */
    protected $taxClassRepository;

    /** @var CatalogHelper $catalogHelper */
    protected $catalogHelper;

    /** @var ProductRepository $productRepository */
    protected $productRepository;

    /** @var AttributeRepository $attributeRepository */
    protected $attributeRepository;

    /** @inheritDoc */
    protected $subordinateWorkersBefore = ['tomagento_integration_product_option'];

    /** @var array $dataToImport */
    protected $dataToImport = [];

    /** @var array $currentData */
    protected $currentData = [];

    /** @var array $currentDefaultData */
    protected $currentDefaultData = [];

    /** @var array $currentVariations */
    protected $currentVariations = [];

    /** @var ProductEntity $currentEntity */
    protected $currentEntity;

    /** @var ProductAttribute $colorAttribute */
    protected $colorAttribute;

    /** @var ProductAttribute $internalColorAttribute */
    protected $internalColorAttribute;

    /** @var ProductAttribute $sizeAttribute */
    protected $sizeAttribute;

    /** @var string $processCategoryPath */
    protected $processCategoryPath;

    /**
     * Save constructor
     *
     * @param Logger                 $logger
     * @param ResourceConnection     $resourceConnection
     * @param FastmagSql             $fastmagSql
     * @param Config                 $config
     * @param JobRepository          $jobRepository
     * @param CategoryRepository    $categoryRepository
     * @param CategoryFactory       $categoryFactory
     * @param CategoryList           $categoryList
     * @param Importer               $importer
     * @param AttributeSetRepository $attributeSetRepository
     * @param SearchCriteriaBuilder  $searchCriteriaBuilder
     * @param Tree                   $tree
     * @param StoresellerRepository  $storesellerRuleRepository
     * @param WebsiteRepository      $websiteRepository
     * @param StoreRepository        $storeRepository
     * @param TaxClassRepository     $taxClassRepository
     * @param CatalogHelper          $catalogHelper
     * @param ProductRepository      $productRepository
     * @param AttributeRepository    $attributeRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        FastmagSql $fastmagSql,
        Config $config,
        JobRepository $jobRepository,
        CategoryRepository $categoryRepository,
        CategoryFactory $categoryFactory,
        CategoryList $categoryList,
        Importer $importer,
        AttributeSetRepository $attributeSetRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        Tree $tree,
        StoresellerRepository $storesellerRuleRepository,
        WebsiteRepository $websiteRepository,
        StoreRepository $storeRepository,
        TaxClassRepository $taxClassRepository,
        CatalogHelper $catalogHelper,
        ProductRepository $productRepository,
        AttributeRepository $attributeRepository
    ) {
        parent::__construct($logger, $resourceConnection, $fastmagSql, $config, $jobRepository);

        $this->categoryRepository = $categoryRepository;
        $this->categoryFactory = $categoryFactory;
        $this->categoryList = $categoryList;
        $this->attributeSetRepository = $attributeSetRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->tree = $tree;
        $this->storesellerRuleRepository = $storesellerRuleRepository;
        $this->websiteRepository = $websiteRepository;
        $this->storeRepository = $storeRepository;
        $this->taxClassRepository = $taxClassRepository;
        $this->catalogHelper = $catalogHelper;
        $this->productRepository = $productRepository;
        $this->attributeRepository = $attributeRepository;

        $this->importer = $importer;
        $this->importer->setBehavior(Import::BEHAVIOR_APPEND)
            ->setEntityCode('catalog_product')
            ->setValidationStrategy(ProcessingErrorAggregatorInterface::VALIDATION_STRATEGY_SKIP_ERRORS)
            ->setAllowedErrorCount(100)
            ->setIgnoreDuplicates(false)
            ->setCategoryPathSeparator('|');
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->notice($exception->getMessage());
            return;
        }

        try {
            $this->getColorSizeAttributesCode();
            $this->checkProcessCategory();
        } catch (JobException $exception) {
            $firstJob = $this->jobs->getItems()[0];

            $firstJob->traceException($exception);
            $this->logger->error(
                __(
                    '[Job #%1] Error on product with Fastmag ID #%2: %3',
                    $firstJob->getId(),
                    $this->getJobEntityId($firstJob),
                    $exception->getMessage()
                )->render()
            );
        }

        foreach ($this->jobs->getItems() as $job) {
            if (!$job->isInError()) {
                try {
                    $this->processJob($job);

                    $job->setStatus(Job::STATUS_OK)
                        ->setProcessedAt(date('Y-m-d H:i:s'));
                } catch (JobException $exception) {
                    $job->traceException($exception);

                    $this->logger->error(
                        __(
                            '[Job #%1] Error on product with Fastmag ID #%2: %3',
                            $job->getId(),
                            $this->getJobEntityId($job),
                            $exception->getMessage()
                        )->render()
                    );
                }
            }
        }

        $this->runImport();

        foreach ($this->jobs->getItems() as $job) {
            if (!$job->isInError()) {
                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $exception) {
                    $job->traceException($exception);
                }
            }
        }
    }

    /**
     * Get the color/size attributes from config and set their code for the jobs
     *
     * @return void
     *
     * @throws JobException
     */
    protected function getColorSizeAttributesCode()
    {
        $colorAttributeId = $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_COLOR);
        $this->colorAttribute = $this->getAttribute($colorAttributeId, 'color');

        $internalColorAttributeId =
            $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_INTERNAL_COLOR);
        $this->internalColorAttribute = $this->getAttribute($internalColorAttributeId, 'internal color');

        $sizeAttributeId = $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_SIZE);
        $this->sizeAttribute = $this->getAttribute($sizeAttributeId, 'size');
    }

    /**
     * Get attribute
     *
     * @param int    $attributeId
     * @param string $label
     *
     * @return ProductAttribute
     *
     * @throws JobException
     */
    protected function getAttribute($attributeId, $label)
    {
        try {
            return $this->attributeRepository->get($attributeId);
        } catch (NoSuchEntityException $exception) {
            throw new JobException(
                __('Attribute "%1", used to save "%2" data, does not exist in Magento', $attributeId, $label)
            );
        }
    }

    /**
     * Get Fastmag process category ID, and create it if it does not exist yet
     *
     * @return void
     *
     * @throws JobException
     */
    protected function checkProcessCategory()
    {
        $this->searchCriteriaBuilder->addFilter(CategoryModel::KEY_NAME, self::PROCESS_CATEGORY_NAME);

        $searchCriteria = $this->searchCriteriaBuilder->create();

        $categories = $this->categoryList->getList($searchCriteria);

        if ($categories->getTotalCount() === 0) {
            $category = $this->categoryFactory->create();

            $category->setName(self::PROCESS_CATEGORY_NAME)
                ->setIncludeInMenu(false)
                ->setIsActive(false)
                ->setParentId(1);

            try {
                $category = $this->categoryRepository->save($category);
            } catch (CouldNotSaveException $exception) {
                throw new JobException(__($exception->getMessage()));
            }
        } else {
            $category = $categories->getItems()[0];
        }

        $this->processCategoryPath = $this->getCategoryFullPath($category);
    }

    /**
     * Get full path of a category
     *
     * @param CategoryModel $category
     *
     * @return string
     */
    protected function getCategoryFullPath($category)
    {
        $categoryTree = $this->tree->loadBreadcrumbsArray($category->getPath());

        $categoryTreePath = [];
        foreach ($categoryTree as $eachCategory) {
            $categoryTreePath[] = $eachCategory['name'];
        }

        return implode('/', $categoryTreePath);
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        $this->currentData = [];
        $this->currentDefaultData = [];

        /** @var ProductEntity $entity */
        $entity = $job->getEntity();
        $this->currentEntity = $entity;

        if ($this->productMustBeSynced($entity)) {
            $this->generateDefaultData();
            $this->generateSimpleProductsData();
            $this->generateConfigurableProductData();
        }
    }

    /**
     * Generate default data for the entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function generateDefaultData()
    {
        $this->currentDefaultData = [
            'attribute_set_code' => $this->getDefaultAttributeSetCode(),
            'categories'         => $this->processCategoryPath,
            'product_websites'   => $this->getWebsites(),
            'tax_class_name'     => $this->getTaxClassName()
        ];

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_CREATION_DATE)) {
            $this->currentDefaultData['created_at'] = $this->currentEntity->getCreationDate();
        }

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_DESCRIPTION)) {
            $this->currentDefaultData['description'] = $this->currentEntity->getDescription();
            $this->currentDefaultData['short_description'] = $this->currentEntity->getDescription();
        }
    }

    /**
     * Returns the default attribute set code for product creation
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getDefaultAttributeSetCode()
    {
        $defaultAttributeSetId = $this->config->getValue(Config::XML_PATH_PRODUCT_IMPORT_ATTRIBUTE_SET_ID);

        if ($defaultAttributeSetId === null) {
            throw new JobException(__(
                'No default attribute set defined in configuration. Please fill the config field before enabling product sync.'
            ));
        }

        try {
            $defaultAttributeSet = $this->attributeSetRepository->get($defaultAttributeSetId);

            return $defaultAttributeSet->getAttributeSetName();
        } catch (NoSuchEntityException $exception) {
            throw new JobException(__(
                'Attribute set with ID #%1 not found in Magento. Please review you config.',
                $defaultAttributeSetId
            ));
        }
    }

    /**
     * Get websites defined in store/seller rules
     *
     * @return string
     */
    protected function getWebsites()
    {
        $result = '';

        $searchCriteria = $this->searchCriteriaBuilder->create();

        $rules = $this->storesellerRuleRepository->getList($searchCriteria);

        $websiteIds = [];
        foreach ($rules->getItems() as $rule) {
            $websiteIds[] = $rule->getWebsiteId();
        }

        // We can't use search criteria logic on website repository
        $websites = $this->websiteRepository->getList();

        foreach ($websites as $website) {
            if (in_array($website->getId(), $websiteIds, true)) {
                $result .= $website->getCode() . ',';
            }
        }

        return trim($result, ',');
    }

    /**
     * Get default tax class name for products
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getTaxClassName()
    {
        $taxClassId = $this->config->getValue(Config::XML_PATH_PRODUCT_PRICE_TAX_CLASS);

        try {
            $taxClass = $this->taxClassRepository->get($taxClassId);

            return $taxClass->getClassName();
        } catch (NoSuchEntityException $exception) {
            throw new JobException(__('Tax class ID #%1 does not exist in Magento', $taxClassId));
        }
    }

    /**
     * Get default Fastmag attributes as an array
     *
     * @return array
     */
    protected function getDefaultAdditionalAttributes()
    {
        $result = [
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_BRAND_CODE    => $this->currentEntity->getBrand(),
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_SUPPLIER_CODE => $this->currentEntity->getSupplier()
        ];

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_FAMILY)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_FAMILY_CODE] = $this->currentEntity->getFamily();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SUBFAMILY)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_SUBFAMILY_CODE] = $this->currentEntity->getSubfamily();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SECTION)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_SECTION_CODE] = $this->currentEntity->getSection();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SEASON)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_SEASON_CODE] = $this->currentEntity->getSeason();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MODEL)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_MODEL_CODE] = $this->currentEntity->getModel();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MATERIAL)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_MATERIAL_CODE] = $this->currentEntity->getMaterial();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_THEME)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_THEME_CODE] = $this->currentEntity->getTheme();
        }

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_TEXTILE)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_WASHING_CODE] = $this->currentEntity->getWashing();
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_CHLORINE_CODE] = $this->currentEntity->getChlorine();
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_IRONING_CODE] = $this->currentEntity->getIroning();
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_DRY_CLEANING_CODE] = $this->currentEntity->getDryCleaning();
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_DRYING_CODE] = $this->currentEntity->getDrying();
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_WATER_CODE] = $this->currentEntity->getWater();
        }

        return $result;
    }

    /**
     * Generate data to integrate for simple products
     *
     * @return void
     *
     * @throws JobException
     */
    protected function generateSimpleProductsData()
    {
        $variations = $this->currentEntity->getVariationsFlat();

        foreach ($variations as $variation) {
            $this->generateSimpleProductData($variation);
        }
    }

    /**
     * Generate data to integrate for simple product
     *
     * @param VariationEntity $variation
     *
     * @return void
     *
     * @throws JobException
     */
    protected function generateSimpleProductData($variation)
    {
        $simpleProductData = [
            'sku'          => $this->generateSimpleProductSku($variation),
            'product_type' => Type::TYPE_SIMPLE,
            'name'         => $this->generateSimpleProductName($variation),
            'weight'       => $variation->getWeight(),
            'visibility'   => 'Not Visible Individually',
            'price'        => $this->getSimpleProductPrice($variation),
            'cost'         => $variation->getBuyingPrice(),
            'url_key'      => $this->generateUrlKey($this->generateSimpleProductName($variation)),
            'qty'          => $this->getSimpleProductStockLevel($variation),
            'additional_attributes' => $this->getSimpleProductAdditionalAttributes($variation)
        ];

        try {
            $this->productRepository->get($simpleProductData['sku']);

            if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_MANAGE_PRODUCT_STATUS)) {
                $simpleProductData['product_online'] = $this->getProductStatus($variation);
            }
        } catch (NoSuchEntityException $e) {
            $simpleProductData['product_online'] = '0';
        }

        $simpleProductData = array_merge($simpleProductData, $this->addRequiredAttributes());

        $this->currentData[] = array_merge($this->currentDefaultData, $simpleProductData);

        $variationConfigurableData = [
            'sku' => $simpleProductData['sku']
        ];

        if (!empty($simpleProductData['additional_attributes'][$this->sizeAttribute->getAttributeCode()])) {
            $variationConfigurableData[$this->sizeAttribute->getAttributeCode()]
                = $simpleProductData['additional_attributes'][$this->sizeAttribute->getAttributeCode()];
        }
        if (!empty($simpleProductData['additional_attributes'][$this->colorAttribute->getAttributeCode()])) {
            $variationConfigurableData[$this->colorAttribute->getAttributeCode()]
                = $simpleProductData['additional_attributes'][$this->colorAttribute->getAttributeCode()];
        }

        $this->currentVariations[] = $variationConfigurableData;

        // TODO check if i18n has actually a value changed from default values
        $i18ns = $variation->getI18ns();
        if ($i18ns !== null && count($i18ns) > 0) {
            foreach ($i18ns as $storeId => $i18n) {
                $i18nData = [
                    'sku'                   => $this->generateSimpleProductSku($variation),
                    'store_view_code'       => $this->getStoreViewCode($storeId),
                    'attribute_set_code'    => $this->getDefaultAttributeSetCode(),
                    'product_type'          => Type::TYPE_SIMPLE,
                    'price'                 => $this->getSimpleProductPrice($variation, $storeId),
                    // Delete all default values unnecessary for store override
                    'categories'            => null,
                    'product_websites'      => null,
                    'tax_class_name'        => null,
                    'additional_attributes' => null
                ];

                $this->currentData[] = array_merge($this->currentDefaultData, $i18nData);
            }
        }
    }

    /**
     * Generate simple product SKU
     *
     * @param VariationEntity $variation
     *
     * @return string
     */
    protected function generateSimpleProductSku($variation)
    {
        $result = $this->currentEntity->getRef() . '-' . $variation->getColor() . '-' . $variation->getSize();

        if ($variation->getColor() === null) {
            $result = $this->currentEntity->getRef() . '-' . $variation->getSize();
        }

        return $result;
    }

    /**
     * Generate simple product name
     *
     * @param VariationEntity $variation
     *
     * @return string
     */
    protected function generateSimpleProductName($variation)
    {
        return trim(
            $this->generateConfigurableProductName() . ' ' . $variation->getColor() . ' ' . $variation->getSize()
        );
    }

    /**
     * Generate configurable product name
     *
     * @return string
     */
    protected function generateConfigurableProductName()
    {
        return trim(
            $this->currentEntity->getDesignation() . ' ' . $this->currentEntity->getDesignationBis()
        );
    }

    /**
     * Get price for a simple product
     *
     * @param VariationEntity $variation
     * @param int|null        $storeId
     *
     * @return float
     */
    protected function getSimpleProductPrice($variation, $storeId = null)
    {
        if ($storeId !== null) {
            $result = $this->getSimpleProductPrice($variation->getI18n($storeId));
        } else {
            $result = $variation->getIndicativePrice();

            $standardPrice = (float)$variation->getStandardPrice();
            $combinationPrice = (float)$variation->getCombinationPrice();
            $ratePrice = (float)$variation->getRatePrice();

            if ($ratePrice > 0) {
                $result = $ratePrice;
            } elseif ($standardPrice > 0) {
                $result = $standardPrice;
            } elseif ($combinationPrice > 0) {
                $result = $combinationPrice;
            }
        }

        return $this->formatPrice($result);
    }

    /**
     * Format price, including or excluding tax
     *
     * @param float $price
     *
     * @return float
     */
    protected function formatPrice($price)
    {
        $result = $price;

        if (!$this->config->isSetFlag(TaxConfig::CONFIG_XML_PATH_PRICE_INCLUDES_TAX)) {
            $vatRate = ($this->currentEntity->getVatRate() ?: 20);

            $result = round($price / (1 + ($vatRate / 100)), 4);
        }

        return $result;
    }

    /**
     * Generate product URL key
     *
     * @param string $productName
     *
     * @return string
     */
    protected function generateUrlKey($productName)
    {
        return strtolower(preg_replace('#[^0-9a-z]+#i', '-', $productName));
    }

    /**
     * Get global stock level, indicated by the stock level and the token is_in_stock for each store
     *
     * @param VariationEntity $variation
     *
     * @return int
     */
    protected function getSimpleProductStockLevel($variation)
    {
        $stockLevel = ($variation->getInStock() ? $variation->getStockLevel() : 0);

        $i18ns = $variation->getI18ns();
        if ($i18ns !== null && count($i18ns) > 0) {
            foreach ($i18ns as $i18n) {
                $stockLevel += $this->getSimpleProductStockLevel($i18n);
            }
        }

        return $stockLevel;
    }

    /**
     * Get Fastmag attributes values for a simple product
     *
     * @param VariationEntity $variation
     *
     * @return array
     */
    protected function getSimpleProductAdditionalAttributes($variation)
    {
        $result = $this->getDefaultAdditionalAttributes();

        $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_PRODUCT_ID_CODE] = $variation->getFastmagId();
        $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_EAN13_CODE] = $variation->getEan13();
        $result[$this->sizeAttribute->getAttributeCode()] = $variation->getSize();

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_USE_COLOR_DESCRIPTION)) {
            $result[$this->colorAttribute->getAttributeCode()] = $variation->getColorDescription();
            $result[$this->internalColorAttribute->getAttributeCode()] = $variation->getColor();
        } else {
            $result[$this->colorAttribute->getAttributeCode()] = $variation->getColor();
        }

        return $result;
    }

    /**
     * Get product enability
     *
     * @param VariationEntity|null $variation
     *
     * @return string
     */
    protected function getProductStatus($variation = null)
    {
        $result = $this->currentEntity->getActive() && $this->currentEntity->getVisibleWeb();

        if ($result && $variation !== null) {
            $result = $variation->getActive();
        }

        return $result ? '1' : '0';
    }

    /**
     * Get store code by store ID
     *
     * @param int $storeId
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getStoreViewCode($storeId)
    {
        try {
            $storeView = $this->storeRepository->getById($storeId);
        } catch (NoSuchEntityException $e) {
            throw new JobException(__('Store #%1 has i18ns set in entity, but does not exist in Magento', $storeId));
        }

        return $storeView->getCode();
    }

    /**
     * Generate data to integrate for configurable product
     *
     * @return void
     */
    protected function generateConfigurableProductData()
    {
        $result = [
            'sku' => $this->currentEntity->getRef(),
            'product_type' => Configurable::TYPE_CODE,
            'name' => trim($this->currentEntity->getDesignation() . ' ' . $this->currentEntity->getDesignationBis()),
            'weight' => $this->currentEntity->getWeight(),
            'visibility' => 'Catalog, Search',
            'url_key' => $this->generateUrlKey(
                trim($this->currentEntity->getDesignation() . ' ' . $this->currentEntity->getDesignation())
            ),
            'additional_attributes' => $this->getDefaultAdditionalAttributes(),
            'configurable_variations' => $this->currentVariations
        ];

        try {
            $this->productRepository->get($result['sku']);

            if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_MANAGE_PRODUCT_STATUS)) {
                $result['product_online'] = $this->getProductStatus();
            }
        } catch (NoSuchEntityException $e) {
            $result['product_online'] = '0';
        }

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_DESCRIPTION)) {
            $result['description'] = $this->currentEntity->getDescription();
            $result['short_description'] = $this->currentEntity->getDescription();
        }

        $result = array_merge($result, $this->addRequiredAttributes());

        $this->currentData[] = array_merge($this->currentDefaultData, $result);
    }

    /**
     * Add required attributes values in imported data to bypass potential exceptions during import
     *
     * @return array
     */
    protected function addRequiredAttributes()
    {
        return [];
    }

    /**
     * Run product importer
     *
     * @throws JobException
     */
    protected function runImport()
    {
        try {
            $currentData = $this->exportCurrentData();

            if (count($currentData) > 0) {
                $this->importer->processImport($currentData);
            }
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }
    }

    /**
     * Export current data in the format expected by the importer
     *
     * @return array
     */
    protected function exportCurrentData()
    {
        if (count($this->currentData) > 0) {
            $index = 0;

            foreach ($this->currentData as $row) {
                if (array_key_exists('additional_attributes', $row) && is_array($row['additional_attributes'])) {
                    $additionnalAttributes = '';

                    foreach ($row['additional_attributes'] as $code => $value) {
                        $additionnalAttributes .= $code . '=' . $value . ',';
                    }

                    $this->currentData[$index]['additional_attributes'] = trim($additionnalAttributes, ',');
                }

                if (array_key_exists('configurable_variations', $row) && is_array($row['configurable_variations'])) {
                    $configurableVariations = '';

                    foreach ($row['configurable_variations'] as $variation) {
                        foreach ($variation as $code => $value) {
                            $configurableVariations .= $code . '=' . $value . ',';
                        }

                        $configurableVariations = trim($configurableVariations, ',') . '|';
                    }

                    $this->currentData[$index]['configurable_variations'] = trim($configurableVariations, '|');
                }

                $index++;
            }
        }

        return $this->currentData;
    }
}
