<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-07-23
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoScheduledInterface;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Jobqueue\ToMagentoFactory;
use Fastmag\Sync\Model\Jobqueue\ToMagentoRepository;
use Fastmag\Sync\Model\Jobqueue\ToMagentoScheduledRepository;
use Fastmag\Sync\Model\Rule\StoresellerRepository;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Worker;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

/**
 * Class Scheduled
 *
 * Worker class used for moving scheduled jobs to the current job queue
 */
class Scheduled extends Worker
{
    /** @inheritDoc */
    protected $code = 'tomagento_move_scheduled';

    /** @var StoresellerRepository $storesellerRepository */
    protected $storesellerRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var ToMagentoFactory $jobFactory */
    protected $jobFactory;

    /** @var ToMagentoRepository $jobRepository */
    protected $jobRepository;

    /** @var ToMagentoScheduledRepository $jobScheduledRepository */
    protected $jobScheduledRepository;

    /**
     * Scheduled constructor.
     *
     * @param Logger                       $logger
     * @param ResourceConnection           $resourceConnection
     * @param FastmagSql                   $fastmagSql
     * @param StoresellerRepository        $storesellerRepository
     * @param SearchCriteriaBuilder        $searchCriteriaBuilder
     * @param ToMagentoFactory             $jobFactory
     * @param ToMagentoRepository          $jobRepository
     * @param ToMagentoScheduledRepository $jobScheduledRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        FastmagSql $fastmagSql,
        StoresellerRepository $storesellerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        ToMagentoFactory $jobFactory,
        ToMagentoRepository $jobRepository,
        ToMagentoScheduledRepository $jobScheduledRepository
    ) {
        parent::__construct($logger, $resourceConnection, $fastmagSql);

        $this->storesellerRepository = $storesellerRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->jobFactory = $jobFactory;
        $this->jobRepository = $jobRepository;
        $this->jobScheduledRepository = $jobScheduledRepository;
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    public function run()
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(ToMagentoScheduledInterface::SCHEDULED_AT, date('Y-m-d H:i:s'))
            ->create();

        $jobsScheduled = $this->jobScheduledRepository->getList($searchCriteria);

        foreach ($jobsScheduled->getItems() as $jobScheduled) {
            try {
                $job = $this->jobRepository->getByUniqueKey(
                    $jobScheduled->getContentId(),
                    $jobScheduled->getJobCode(),
                    10
                );

                $job->setProcessedAt(null)
                    ->setContent($job->getContent());
            } catch (NoSuchEntityException $exception) {
                $job = $this->jobFactory->create([
                    ToMagentoInterface::CONTENT_ID => $jobScheduled->getContentId(),
                    ToMagentoInterface::JOB_CODE   => $jobScheduled->getJobCode(),
                    ToMagentoInterface::CREATED_AT => $jobScheduled->getCreatedAt(),
                    ToMagentoInterface::CONTENT    => $jobScheduled->getContent(),
                    ToMagentoInterface::PRIORITY   => 10
                ]);
            }

            try {
                $this->jobRepository->save($job);

                $this->jobScheduledRepository->delete($jobScheduled);
            } catch (Exception $exception) {
                throw new ProcessException(__($exception->getMessage()));
            }
        }

        $this->deleteByShops();

        $this->setLastExecutionDate();
    }

    /**
     * Delete scheduled jobs for shops set in store/seller rules
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function deleteByShops()
    {
        $shopsList = $this->getShopsList();

        if ($shopsList !== '') {
            $searchCriteria = $this->searchCriteriaBuilder
                ->addFilter(ToMagentoScheduledInterface::FASTMAG_SHOP, $shopsList, 'in')
                ->create();

            $jobsToDelete = $this->jobScheduledRepository->getList($searchCriteria);

            foreach ($jobsToDelete->getItems() as $job) {
                try {
                    $this->jobScheduledRepository->delete($job);
                } catch (CouldNotDeleteException $exception) {
                    throw new ProcessException(__($exception->getMessage()));
                }
            }
        }
    }

    /**
     * Get the shops list from the Storeseller rules repository
     *
     * @return string
     */
    protected function getShopsList()
    {
        $shopsList = [];

        $searchCriteria = $this->searchCriteriaBuilder->create();
        $rulesCollection = $this->storesellerRepository->getList($searchCriteria);

        foreach ($rulesCollection->getItems() as $rule) {
            $shopsList[] = '\'' . $rule->getFastmagShop() . '\'';
        }

        return implode(', ', $shopsList);
    }
}
