<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-06-08
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\RemoteSync\Sync;

use Fastmag\Sync\Api\Jobqueue\ToMagentoScheduledRepositoryInterface;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Jobqueue\ToMagentoScheduledFactory;
use Fastmag\Sync\Model\Rule\StoresellerRepository;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Worker\RemoteSync\Sync;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;

/**
 * Class Scheduled
 *
 * Sync worker class used for sync all jobs which have to be processed in a scheduled time
 */
class Scheduled extends Sync
{
    /** @inheritDoc */
    protected $code = 'remotesync_sync_scheduled';

    /** @var StoresellerRepository $storesellerRepository */
    protected $storesellerRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var ToMagentoScheduledFactory $jobFactory */
    protected $jobFactory;

    /** @var ToMagentoScheduledRepositoryInterface $jobRepository */
    protected $jobRepository;

    /**
     * Instant constructor
     *
     * @param Logger                                $logger
     * @param ResourceConnection                    $resourceConnection
     * @param FastmagSql                            $fastmagSql
     * @param StoresellerRepository                 $storesellerRepository
     * @param SearchCriteriaBuilder                 $searchCriteriaBuilder
     * @param ToMagentoScheduledFactory             $jobFactory
     * @param ToMagentoScheduledRepositoryInterface $jobRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        FastmagSql $fastmagSql,
        StoresellerRepository $storesellerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        ToMagentoScheduledFactory $jobFactory,
        ToMagentoScheduledRepositoryInterface $jobRepository
    ) {
        parent::__construct($logger, $resourceConnection, $fastmagSql);

        $this->storesellerRepository = $storesellerRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->jobFactory = $jobFactory;
        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     */
    protected function saveJob($event)
    {
        $createdAtDate = $this->getUtcDateTimeFromFastmag($event['date_maj']);
        $scheduledAtDate = $this->getUtcDateTimeFromFastmag($event['date_a_traiter']);

        $job = $this->jobFactory->create();
        $job->setContentId($event['identifiant'])
            ->setJobCode($this->generateJobCode($event['job']))
            ->setFastmagShop($event['code_mag'])
            ->setContent($event['commentaire'])
            ->setCreatedAt($event['date_maj'])
            ->setScheduledAt($scheduledAtDate);
        $this->jobRepository->save($job);

        $newLastExecution = $createdAtDate;

        $this->logger->debug('Last Id : ' . $event['identifiant'] . ' - Last date : ' . $newLastExecution);
    }

    /**
     * @inheritDoc
     */
    protected function getSqlCondition()
    {
        $shopsList = $this->getShopsList();
        $shopsCondition = '';
        if ($shopsList !== '') {
            $shopsCondition = ' AND code_mag IN (' . $shopsList . ')';
        }

        return '(date_a_traiter IS NOT NULL AND date_a_traiter > NOW())' . $shopsCondition;
    }

    /**
     * Get the shops list from the Storeseller rules repository
     *
     * @return string
     */
    protected function getShopsList()
    {
        $shopsList = [];

        $searchCriteria = $this->searchCriteriaBuilder->create();
        $rulesCollection = $this->storesellerRepository->getList($searchCriteria);

        foreach ($rulesCollection->getItems() as $rule) {
            $shopsList[] = '\'' . $rule->getFastmagShop() . '\'';
        }

        return implode(', ', $shopsList);
    }
}
