<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-07-23
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer;

use Fastmag\Sync\Api\CustomerRepositoryInterface as SyncedCustomerRepository;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\CustomerFactory as CustomerEntityFactory;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer\Address as AddressHydrationWorker;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Newsletter\Model\Subscriber;

/**
 * Class Save
 *
 * Hydration class used for inserting or updating customers from Magento to Fastmag
 */
class Save extends Customer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_customer_save';

    /** @var Subscriber $subscriber */
    protected $subscriber;

    /** @var AddressHydrationWorker $addressHydrationWorker */
    protected $addressHydrationWorker;

    /** @inheritDoc */
    protected $subordinateWorkersAfter = ['tofastmag_hydration_customer_address'];

    /**
     * Customer constructor.
     *
     * @param Logger                      $logger
     * @param ResourceConnection          $resourceConnection
     * @param FastmagSql                  $fastmagSql
     * @param Config                      $config
     * @param SyncedCustomerRepository    $syncedCustomerRepository
     * @param SearchCriteriaBuilder       $searchCriteriaBuilder
     * @param CustomerRepositoryInterface $customerRepository
     * @param JobRepository               $jobRepository
     * @param CustomerEntityFactory       $customerEntityFactory
     * @param Subscriber                  $subscriber
     * @param AddressHydrationWorker      $addressHydrationWorker
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        FastmagSql $fastmagSql,
        Config $config,
        SyncedCustomerRepository $syncedCustomerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        CustomerRepositoryInterface $customerRepository,
        JobRepository $jobRepository,
        CustomerEntityFactory $customerEntityFactory,
        Subscriber $subscriber,
        AddressHydrationWorker $addressHydrationWorker
    ) {
        parent::__construct(
            $logger,
            $resourceConnection,
            $fastmagSql,
            $config,
            $syncedCustomerRepository,
            $searchCriteriaBuilder,
            $customerRepository,
            $jobRepository,
            $customerEntityFactory
        );

        $this->subscriber = $subscriber;
        $this->addressHydrationWorker = $addressHydrationWorker;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        $entities = false;

        try {
            $entities = $this->getDataFromMagento();
        } catch (ProcessException $exception) {
            foreach ($this->jobs->getItems() as $job) {
                $job->traceException($exception);
            }
        }

        if (is_array($entities)) {
            foreach ($entities as $customerId => $entity) {
                $job = $this->getJob($customerId);

                if ($job !== null) {
                    $job->setHydratedData($entity->export())
                        ->setEntity($entity);
                }
            }
        }

        foreach ($this->jobs->getItems() as $job) {
            try {
                $this->jobRepository->save($job);
            } catch (CouldNotSaveException $exception) {
                $job->traceException($exception);
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @return CustomerEntity[]
     *
     * @throws ProcessException
     */
    protected function getDataFromMagento()
    {
        $this->entities = [];
        $customersIds = $this->jobs->getColumnValues(Job::CONTENT_ID);

        foreach ($customersIds as $customerId) {
            $customerEntity = $this->customerEntityFactory->create();
            $customerEntity->setMagentoId($customerId);
            $this->entities[$customerId] = $customerEntity;
        }

        $this->getSyncedCustomers($customersIds);
        $this->getMagentoCustomersData($customersIds);

        return $this->entities;
    }

    /**
     * Get Magento data for the current jobs
     *
     * @param int[] $customersIds
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getMagentoCustomersData($customersIds)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('entity_id', $customersIds, 'in')
            ->create();

        try {
            $customersList = $this->customerRepository->getList($searchCriteria);

            foreach ($customersList->getItems() as $customer) {
                $currentEntity = $this->entities[$customer->getId()];

                $currentEntity->setGender($customer->getGender())
                    ->setPrefix($customer->getPrefix())
                    ->setLastname($customer->getLastname())
                    ->setFirstname($customer->getFirstname())
                    ->setEmailAddress($customer->getEmail())
                    ->setCreatedAt($customer->getCreatedAt())
                    ->setStoreId($customer->getStoreId())
                    ->setDob($customer->getDob());

                $checkSubscriber = $this->subscriber->loadByCustomerId($customer->getId());
                $currentEntity->setSubscribeNewsletter($checkSubscriber->isSubscribed());
            }
        } catch (LocalizedException $exception) {
            throw new ProcessException(__(
                'Error when hydrating customers. Message: %1. Customers IDs: %2',
                $exception->getMessage(),
                implode(', ', $customersIds)
            ));
        }
    }
}
