<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-13
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration;

use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Data\OrderInterfaceFactory as SyncedOrderFactory;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Data\Rule\PaymentcodeInterface as PaymentcodeRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\OrderRepositoryInterface as SyncedOrderRepository;
use Fastmag\Sync\Api\Rule\OrdertransactionRepositoryInterface as OrdertransactionRuleRepository;
use Fastmag\Sync\Api\Rule\PaymentcodeRepositoryInterface as PaymentcodeRuleRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\OrderFactory as OrderEntityFactory;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration;
use Magento\Catalog\Model\Product\Type;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Sales\Api\Data\OrderInterface as OrderObject;
use Magento\Sales\Api\OrderRepositoryInterface as OrderRepository;
use Magento\Store\Model\ScopeInterface;

/**
 * Class Order
 *
 * Hydration class used for inserting or updating orders from Magento to Fastmag
 */
class Order extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_order';

    /** @var OrderRepository $orderRepository */
    protected $orderRepository;

    /** @var SyncedOrderRepository $syncedOrderRepository */
    protected $syncedOrderRepository;

    /** @var SyncedOrderFactory $syncedOrderFactory */
    protected $syncedOrderFactory;

    /** @var OrdertransactionRuleRepository $ordertransactionRuleRepository */
    protected $ordertransactionRuleRepository;

    /** @var PaymentcodeRuleRepository $paymentcodeRuleRepository */
    protected $paymentcodeRuleRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /** @var OrderEntityFactory $orderEntityFactory */
    protected $orderEntityFactory;

    /** @var Job $currentJob */
    protected $currentJob;

    /** @var OrdertransactionRule[] $ordertransactionRules */
    protected $ordertransactionRules;

    /** @var PaymentcodeRule[] $paymentcodeRules */
    protected $paymentcodeRules;

    /** @inheritDoc */
    protected $subordinateWorkersAfter = ['tofastmag_hydration_order_customer', 'tofastmag_hydration_order_address'];

    /**
     * Order constructor.
     *
     * @param Logger                         $logger
     * @param ResourceConnection             $resourceConnection
     * @param FastmagSql                     $fastmagSql
     * @param Config                         $config
     * @param OrderRepository                $orderRepository
     * @param SyncedOrderRepository          $syncedOrderRepository
     * @param SyncedOrderFactory             $syncedOrderFactory
     * @param OrdertransactionRuleRepository $ordertransactionRuleRepository
     * @param PaymentcodeRuleRepository      $paymentcodeRuleRepository
     * @param SearchCriteriaBuilder          $searchCriteriaBuilder
     * @param JobRepository                  $jobRepository
     * @param OrderEntityFactory             $orderEntityFactory
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        FastmagSql $fastmagSql,
        Config $config,
        OrderRepository $orderRepository,
        SyncedOrderRepository $syncedOrderRepository,
        SyncedOrderFactory $syncedOrderFactory,
        OrdertransactionRuleRepository $ordertransactionRuleRepository,
        PaymentcodeRuleRepository $paymentcodeRuleRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        JobRepository $jobRepository,
        OrderEntityFactory $orderEntityFactory
    ) {
        parent::__construct($logger, $resourceConnection, $fastmagSql, $config);

        $this->orderRepository = $orderRepository;
        $this->syncedOrderRepository = $syncedOrderRepository;
        $this->syncedOrderFactory = $syncedOrderFactory;
        $this->ordertransactionRuleRepository = $ordertransactionRuleRepository;
        $this->paymentcodeRuleRepository = $paymentcodeRuleRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->jobRepository = $jobRepository;
        $this->orderEntityFactory = $orderEntityFactory;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        $this->getRules();

        foreach ($this->jobs->getItems() as $job) {
            $this->currentJob = $job;

            try {
                $entity = $this->getDataFromMagento();

                $job->setHydratedData($entity->export())
                    ->setEntity($entity);
            } catch (JobException $exception) {
                $job->traceException($exception);

                $this->logger->error(
                    __(
                        '[Job #%1] Error on customer with Magento ID #%2: %3',
                        $job->getId(),
                        $this->getOrderIdFromJobContentId($job->getContentId()),
                        $exception->getMessage()
                    )->render()
                );
            }

            try {
                $this->jobRepository->save($job);
            } catch (CouldNotSaveException $exception) {
                $job->traceException($exception);
            }
        }
    }

    /**
     * Get rules
     *
     * @return void
     */
    protected function getRules()
    {
        $criteria = $this->searchCriteriaBuilder->create();

        $this->ordertransactionRules = $this->ordertransactionRuleRepository->getList($criteria)->getItems();
        $this->paymentcodeRules = $this->paymentcodeRuleRepository->getList($criteria)->getItems();
    }

    /**
     * @inheritDoc
     *
     * @return OrderEntity
     *
     * @throws JobException
     */
    protected function getDataFromMagento()
    {
        $orderEntity = $this->orderEntityFactory->create();

        [$orderId, $status] = $this->getOrderDataFromJobContentId($this->currentJob->getContentId());

        $orderEntity->setMagentoId($orderId)
            ->setStatus($status);

        $order = $this->orderRepository->get($orderId);
        $orderEntity->setCustomerId($order->getCustomerId());

        $paymentMethod = $this->getOrderPaymentMethod($order);
        $orderEntity->setPaymentCode($this->getFastmagPaymentCode($paymentMethod));

        $transactionType = $this->getTransactionType($paymentMethod, $status);

        if ($transactionType === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE) {
            try {
                $orderEntity->setLastTransaction($this->syncedOrderRepository->getLastSyncByOrderId($orderId));

                if ($orderEntity->getLastTransaction()->getTransactionType()
                    === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION
                ) {
                    $transactionType = OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE;
                }
            } catch (LocalizedException $exception) {
                // Do nothing
            }
        } elseif ($transactionType === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_CANCELLATION) {
            try {
                $orderEntity->setLastTransaction($this->syncedOrderRepository->getLastSyncByOrderId($orderId, []));
            } catch (LocalizedException $exception) {
                $transactionType = OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_NONE;
            }
        }

        if ($transactionType === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_NONE) {
            throw new JobException(__(
                'No transaction type found for order #%1. Order status: %2 - Payment method: %3',
                $order->getIncrementId(),
                $status,
                $paymentMethod
            ));
        }

        $orderEntity->setTransactionType($transactionType);
        $orderEntity = $this->setFastmagHeaders($orderEntity, $order->getStoreId());

        $orderEntity->setEdiFailed($this->checkTransactionDuplicate($orderId, $orderEntity->getTransactionType()))
            ->setIsExcludedTax($this->isOrderExcludingTax($order, $orderEntity->getTransactionType()))
            ->setTotalQty($this->getTotalQuantity($order, $orderEntity->getTransactionType()));

        $items = $this->getOrderItems($order, $orderEntity->getTransactionType());

        foreach ($items as $item) {
            $orderEntity->addItem($item);
        }

        $this->getFastmagProductData($orderEntity);

        $shippingData = $this->getShippingData($order, $orderEntity->getTransactionType());

        $orderEntity->setShippingRate($shippingData['shipping_rate'])
            ->setShippingDiscount($shippingData['shipping_discount'])
            ->setPaymentId($this->getPaymentId($order, $orderEntity->getTransactionType()));

        return $orderEntity;
    }

    /**
     * Retrieve order payment method
     *
     * @param OrderObject $order
     *
     * @return string
     */
    public function getOrderPaymentMethod($order)
    {
        $result = '*';

        if ($order->getPayment()) {
            $result = $order->getPayment()->getMethod();
        }

        return $result;
    }

    /**
     * Get most specific order/transaction rule given the payment method and the status of the order.
     *
     * @param string $paymentMethod
     * @param string $orderStatus
     *
     * @return string
     */
    protected function getTransactionType($paymentMethod, $orderStatus)
    {
        $result = OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_NONE;
        $bestRule = null;

        foreach ($this->ordertransactionRules as $rule) {
            $rulePaymentMethod = $rule->getPaymentMethod();
            $ruleOrderStatus = $rule->getOrderStatus();

            if ($bestRule === null && $rulePaymentMethod === '*' && $ruleOrderStatus === '*') {
                $bestRule = $rule;
            }
            if ($bestRule->getOrderStatus() === '*'
                && $rulePaymentMethod === '*'
                && $ruleOrderStatus === $orderStatus
            ) {
                $bestRule = $rule;
            }
            if ($rulePaymentMethod === $paymentMethod && $ruleOrderStatus === $orderStatus) {
                $bestRule = $rule;
            }
        }

        if ($bestRule && $bestRule->getTransactionType() !== null) {
            $result = $bestRule->getTransactionType();
        }

        if ($bestRule !== null) {
            $this->logger->debug(
                __(
                    'Best order/transaction rule found for job "%1". Payment method: %2". Rule #%3, transaction type: %4',
                    $this->currentJob->getContentId(),
                    $paymentMethod,
                    $bestRule->getId(),
                    $result
                )->render()
            );
        } else {
            $this->logger->debug(
                __(
                    'No order/transaction rule found for job  "%1". Payment method: %2"',
                    $this->currentJob->getContentId(),
                    $paymentMethod
                )->render()
            );
        }

        return $result;
    }

    /**
     * Set Fastmag chain, shop, seller and stock according to config
     *
     * @param OrderEntity $orderEntity
     * @param int         $storeId
     *
     * @return OrderEntity
     *
     * @throws JobException
     */
    protected function setFastmagHeaders($orderEntity, $storeId)
    {
        $orderEntity->setFastmagChain(
            $this->config->getValue(Config::XML_PATH_ORDER_WORKFLOW_CHAIN, ScopeInterface::SCOPE_STORES, $storeId)
        );
        $orderEntity->setFastmagShop(
            $this->config->getValue(Config::XML_PATH_ORDER_WORKFLOW_SHOP, ScopeInterface::SCOPE_STORES, $storeId)
        );
        $orderEntity->setFastmagSeller(
            $this->config->getValue(Config::XML_PATH_ORDER_WORKFLOW_SELLER, ScopeInterface::SCOPE_STORES, $storeId)
        );

        $orderEntity->setFastmagReferenceStock(
            $this->config->getValue(
                Config::XML_PATH_INVENTORY_FASTMAG_STOCK_REFERENCE_STOCK,
                ScopeInterface::SCOPE_STORES,
                $storeId
            )
        );

        $orderEntity->setFastmagAlternativeStocks(
            $this->config->getValue(
                Config::XML_PATH_INVENTORY_FASTMAG_STOCK_ALTERNATIVE_STOCKS,
                ScopeInterface::SCOPE_STORES,
                $storeId
            )
        );

        if (!$orderEntity->getFastmagChain()
            || !$orderEntity->getFastmagShop()
            || !$orderEntity->getFastmagSeller()
            || !$orderEntity->getFastmagStocks()
        ) {
            throw new JobException(__('Missing chain/shop/seller/stock configuration, unable to sync order.'));
        }

        return $orderEntity;
    }

    /**
     * Check if the order with $orderId has not been already sent to Fastmag.
     * Workaround to avoid duplication in case of Fastmag EDI crash during order sync.
     *
     * @param int    $orderId
     * @param string $transactionType
     *
     * @return bool
     *
     * @throws JobException
     */
    protected function checkTransactionDuplicate($orderId, $transactionType)
    {
        $result = false;

        if (in_array(
            $transactionType,
            [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION
            ],
            true
        )) {
            $isOrderSyncedInMagento = (bool)$this->syncedOrderRepository->getListByOrderId($orderId)->getTotalCount();

            try {
                $sql = 'SELECT v.Vente AS transaction_id, v.Date AS date, v.Heure AS hour, v.CodeMag AS target_shop
                    FROM ventes
                    WHERE v.Nature = ' . $this->getFastmagSqlConnection()->escape($transactionType) . '
                        AND v.InfosComp LIKE ' . $this->getFastmagSqlConnection()->escape($orderId) . '
                    ORDER BY v.Vente DESC
                    LIMIT 1';

                $row = $this->getFastmagSqlConnection()->get($sql);
            } catch (NoConnectionException $exception) {
                throw new JobException(__(
                    'Error when trying to check order ID in Fastmag for duplicates. Message: %1. Order ID: %2 - Transaction type: %3',
                    $exception->getMessage(),
                    $orderId,
                    $transactionType
                ));
            }

            if (!$isOrderSyncedInMagento && count($row) > 0) {
                $result = true;

                $fastmagTransactionData = $row[0];
                $syncedOrder = $this->syncedOrderFactory->create();

                $syncedOrder->setOrderId($orderId)
                    ->setTransactionId($fastmagTransactionData['transaction_id'])
                    ->setTransactionType($transactionType)
                    ->setState('OK')
                    ->setTargetShop($fastmagTransactionData['target_shop'])
                    ->setRequestAt($fastmagTransactionData['date'] . ' ' . $fastmagTransactionData['hour'])
                    ->setResultAt($fastmagTransactionData['date'] . ' ' . $fastmagTransactionData['hour']);

                try {
                    $this->syncedOrderRepository->save($syncedOrder);
                } catch (CouldNotSaveException $exception) {
                    throw new JobException(__(
                        'Error when trying to save synced order during first check. Message: %1. Order ID: %2 - Transaction type: %3',
                        $exception->getMessage(),
                        $orderId,
                        $transactionType
                    ));
                }
            }
        }

        return $result;
    }

    /**
     * Check if order is including or excluding tax
     *
     * @param OrderObject $order
     * @param string      $transactionType
     *
     * @return bool
     *
     * @todo: find a better way to find out if an order is taxed or not
     */
    protected function isOrderExcludingTax($order, $transactionType)
    {
        $result = false;

        if (in_array(
            $transactionType,
            [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_ORDER
            ],
            true
        )) {
            $result = $order->getTaxAmount() <= 0;
        }

        return $result;
    }

    /**
     * Get order data (ID, state and status) from job content ID
     *
     * @param int $contentId
     *
     * @return array
     */
    protected function getOrderDataFromJobContentId($contentId)
    {
        return explode('_', $contentId);
    }

    /**
     * Get order ID from job content ID
     *
     * @param int $contentId
     *
     * @return int
     */
    protected function getOrderIdFromJobContentId($contentId)
    {
        $orderData = $this->getOrderDataFromJobContentId($contentId);

        return $orderData[0];
    }

    /**
     * Retrieve items data
     *
     * @param OrderObject $order
     * @param string      $transactionType
     *
     * @return array
     */
    protected function getOrderItems($order, $transactionType)
    {
        $result = [];

        if (!in_array(
            $transactionType,
            [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_NONE
            ],
            true
        )) {
            foreach ($order->getItems() as $item) {
                if ($item->getProductType() === Type::TYPE_SIMPLE) {
                    $result[] = $item;
                }
            }
        }

        return $result;
    }

    /**
     * Retrieve total quantity of the order (ordered or cancelled)
     *
     * @param OrderObject $order
     * @param string      $transactionType
     *
     * @return int
     */
    protected function getTotalQuantity($order, $transactionType)
    {
        $result = 0;

        if ($transactionType === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_CANCELLATION) {
            foreach ($order->getItems() as $item) {
                $result += $item->getQtyCanceled();
            }
        } else {
            $result = $order->getTotalQtyOrdered();
        }

        return $result;
    }

    /**
     * Get basic data from Fastmag
     *
     * @param OrderEntity $orderEntity
     *
     * @return array
     *
     * @throws JobException
     */
    protected function getFastmagProductData($orderEntity)
    {
        $result = $itemsFastmagIds = [];

        $items = $orderEntity->getItems();

        foreach ($items as $itemId => $item) {
            $itemsFastmagIds[$item->getFastmagId()] = $itemId;
        }

        try {
            $sql = 'SELECT Produit AS product_id, BarCode AS barcode, Taille AS size, Couleur AS color
                FROM produits
                WHERE Produit IN ' . $this->getFastmagSqlConnection()->escape(array_keys($itemsFastmagIds));

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (NoConnectionException $exception) {
            throw new JobException(__(
                'Error when trying to check order\'s items product ids in Fastmag. Message: %1. Product IDs: %3',
                $exception->getMessage(),
                array_keys($itemsFastmagIds)
            ));
        }

        if (count($rows) < count($items)) {
            throw new JobException(__(
                'Unable to find corresponding products in Fastmag for some items of the order #%1',
                $orderEntity->getMagentoId()
            ));
        }

        foreach ($rows as $row) {
            $itemEntity = $orderEntity->getItemByFastmagId($row['product_id']);

            if ($itemEntity !== null) {
                $itemEntity->setFastmagBarcode($row['barcode'])
                    ->setFastmagSize($row['size'])
                    ->setFastmagColor($row['barcode']);
            }
        }

        return $result;
    }

    /**
     * Get shipping rate data
     *
     * @param OrderObject $order
     * @param string      $transactionType
     *
     * @return array
     */
    protected function getShippingData($order, $transactionType)
    {
        $result = [];

        if (!in_array(
            $transactionType,
            [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_CANCELLATION,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_NONE
            ],
            true
        )) {
            $result = [
                'shipping_rate'     => $order->getBaseShippingInclTax(),
                'shipping_discount' => $order->getBaseShippingDiscountAmount()
            ];
        }
        
        return $result;
    }

    /**
     * Get payment data
     *
     * @param OrderObject $order
     * @param string      $transactionType
     *
     * @return string
     */
    protected function getPaymentId($order, $transactionType)
    {
        $result = '';

        if (in_array(
            $transactionType,
            [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_ORDER
            ],
            true
        )) {
            $payment = $order->getPayment();

            if ($payment !== null) {
                $result = $payment->getEntityId();

                if ($payment->getCcTransId()) {
                    $result = $payment->getCcTransId();
                } elseif ($payment->getLastTransId()) {
                    $result = $payment->getLastTransId();
                }
            }
        }

        return $result;
    }

    /**
     * Get Fastmag payment code
     *
     * @param string $paymentMethod
     *
     * @return string
     */
    protected function getFastmagPaymentCode($paymentMethod)
    {
        $result = '';
        $bestRule = null;

        foreach ($this->paymentcodeRules as $rule) {
            if ($rule->getPaymentMethod() === $paymentMethod) {
                $bestRule = $rule;
            }
            if ($bestRule === null && $rule->getPaymentMethod() === '*') {
                $bestRule = $rule;
            }
        }

        if ($bestRule && $bestRule->getFastmagCode() !== null) {
            $result = $bestRule->getFastmagCode();
        }

        if ($bestRule !== null) {
            $this->logger->debug(
                __(
                    'Best payment/code rule found for job "%1". Payment method: "%2". Rule #%3, Fastmag code: %4',
                    $this->currentJob->getContentId(),
                    $paymentMethod,
                    $bestRule->getId(),
                    $result
                )->render()
            );
        } else {
            $this->logger->debug(
                __(
                    'No payment/code rule found for job "%1". Payment method: "%2"',
                    $this->currentJob->getContentId(),
                    $paymentMethod
                )->render()
            );
        }

        return $result;
    }
}
