<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-06
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Entity\ToMagento\Product as ProductEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Product\Variation as VariationEntity;
use Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;
use Magento\Catalog\Api\ProductAttributeRepositoryInterface as AttributeRepository;
use Magento\Catalog\Api\Data\ProductAttributeInterface as Attribute;
use Magento\Catalog\Model\Product\Attribute\OptionManagement as AttributeOptionManager;
use Magento\Eav\Api\Data\AttributeOptionInterface as AttributeOption;
use Magento\Eav\Api\Data\AttributeOptionInterfaceFactory as AttributeOptionFactory;
use Magento\Eav\Api\Data\AttributeOptionLabelInterfaceFactory as AttributeOptionLabelFactory;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\NoSuchEntityException;

/**
 * Class Option
 *
 * Integration worker setting attributes option before saving products
 */
class Option extends Product
{
    /** @inheritDoc */
    protected $code = 'tomagento_integration_product_option';

    /** @var AttributeRepository $attributeRepository */
    protected $attributeRepository;

    /** @var AttributeOptionManager $attributeOptionManager */
    protected $attributeOptionManager;

    /** @var AttributeOptionFactory $attributeOptionFactory */
    protected $attributeOptionFactory;

    /** @var AttributeOptionLabelFactory $attributeOptionLabelFactory */
    protected $attributeOptionLabelFactory;

    /** @var bool $useColorDescrption */
    protected $useColorDescrption;

    /**
     * Customer constructor.
     *
     * @param Logger                      $logger
     * @param ResourceConnection          $resourceConnection
     * @param FastmagSql                  $fastmagSql
     * @param Config                      $config
     * @param JobRepository               $jobRepository
     * @param AttributeRepository         $attributeRepository
     * @param AttributeOptionManager      $attributeOptionManager
     * @param AttributeOptionFactory      $attributeOptionFactory
     * @param AttributeOptionLabelFactory $attributeOptionLabelFactory
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        FastmagSql $fastmagSql,
        Config $config,
        JobRepository $jobRepository,
        AttributeRepository $attributeRepository,
        AttributeOptionManager $attributeOptionManager,
        AttributeOptionFactory $attributeOptionFactory,
        AttributeOptionLabelFactory $attributeOptionLabelFactory
    ) {
        parent::__construct($logger, $resourceConnection, $fastmagSql, $config, $jobRepository);

        $this->attributeRepository = $attributeRepository;
        $this->attributeOptionManager = $attributeOptionManager;
        $this->attributeOptionFactory = $attributeOptionFactory;
        $this->attributeOptionLabelFactory = $attributeOptionLabelFactory;

        $this->useColorDescrption =
            $this->config->isSetFlag(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_USE_COLOR_DESCRIPTION);
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        /** @var ProductEntity $entity */
        $entity = $job->getEntity();

        if ($this->productMustBeSynced($entity)) {
            $entityFields = [
                'brand'        => 'fastmag_brand',
                'supplier'     => 'fastmag_supplier',
                'family'       => 'fastmag_family',
                'subfamily'    => 'fastmag_subfamily',
                'section'      => 'fastmag_section',
                'season'       => 'fastmag_season',
                'material'     => 'fastmag_material',
                'model'        => 'fastmag_model',
                'theme'        => 'fastmag_theme',
                'washing'      => 'fastmag_washing',
                'chlorine'     => 'fastmag_chlorine',
                'ironing'      => 'fastmag_ironing',
                'dry_cleaning' => 'fastmag_dry_cleaning',
                'drying'       => 'fastmag_drying',
                'water'        => 'fastmag_water',
            ];

            foreach ($entityFields as $fieldCode => $attributeCode) {
                $defaultValue = $entity->getData($fieldCode);

                $this->saveOption($attributeCode, $defaultValue, $entity->getDataI18n($fieldCode));
            }

            foreach ($entity->getVariationsFlat() as $variation) {
                $this->saveColorSizeOptions($variation);
            }
        }
    }

    /**
     * Create or update option value with all the i18nized labels
     *
     * @todo Get options labels to check if there are differencies with the labels we want to save
     *
     * @param int|string $attributeCode
     * @param string     $defaultValue
     * @param array      $i18nValues
     * @param string     $label
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveOption($attributeCode, $defaultValue, $i18nValues = [], $label = null)
    {
        try {
            $attribute = $this->attributeRepository->get($attributeCode);
        } catch (NoSuchEntityException $exception) {
            if ($label !== null) {
                throw new JobException(
                    __('Attribute "%1", used to save "%2" data, does not exist in Magento', $attributeCode, $label)
                );
            } else {
                throw new JobException(__('Attribute "%1" does not exist in Magento', $attributeCode));
            }
        }

        if (!empty($defaultValue)) {
            $option = $this->getOption($attribute, $defaultValue);
            $optionLabels = $this->getStoreLabels($i18nValues);

            if (!$option) {
                $option = $this->attributeOptionFactory->create();
                $option->setLabel($defaultValue)
                    ->setStoreLabels($optionLabels);

                try {
                    $this->attributeOptionManager->add($attributeCode, $option);
                } catch (Exception $exception) {
                    throw new JobException(
                        __('Exception when adding option "%1" of the attribute "%2"', $defaultValue, $attributeCode)
                    );
                }
            } else {
                $option->setStoreLabels($optionLabels);

                try {
                    $this->attributeOptionManager->update(
                        $attributeCode,
                        $attribute->getSource()->getOptionId($defaultValue),
                        $option
                    );
                } catch (Exception $exception) {
                    throw new JobException(__(
                        'Exception when updating option "%1" of the attribute "%2" with these labels: %3',
                        $defaultValue,
                        $attributeCode,
                        implode(', ', array_values($i18nValues))
                    ));
                }
            }
        }
    }

    /**
     * Check if option already exists for attribute
     *
     * @param Attribute $attribute
     * @param string    $defaultValue
     *
     * @return AttributeOption|false
     */
    protected function getOption($attribute, $defaultValue)
    {
        $result = false;

        if ($attribute->getBackendType() === 'int' && $attribute->getFrontendInput() === 'select') {
            $options = $attribute->getOptions();

            foreach ($options as $option) {
                if ($option->getLabel() === $defaultValue) {
                    $result = $option;
                    break;
                }
            }
        }

        return $result;
    }

    /**
     * Get array of AttributeOptionLabel object to be saved with the option
     *
     * @param array $i18nValues
     */
    protected function getStoreLabels($i18nValues)
    {
        $result = null;

        foreach ($i18nValues as $storeId => $value) {
            $optionLabel = $this->attributeOptionLabelFactory->create();
            $optionLabel->setStoreId($storeId)
                ->setLabel($value);

            $result[] = $optionLabel;
        }

        return $result;
    }

    /**
     * Save color and size options given the config of a product's variation
     *
     * @param VariationEntity $entity
     *
     * @throws JobException
     */
    protected function saveColorSizeOptions($entity)
    {
        $colorAttributeId = $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_COLOR);

        if ($this->useColorDescrption) {
            $this->saveOption(
                $colorAttributeId,
                $entity->getData('color_description'),
                $entity->getDataI18n('color_description')
            );

            $this->saveOption(
                $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_INTERNAL_COLOR),
                $entity->getData('color'),
                $entity->getDataI18n('color')
            );
        } else {
            $this->saveOption($colorAttributeId, $entity->getData('color'), $entity->getDataI18n('color'));
        }

        $this->saveOption(
            $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_SIZE),
            $entity->getData('size'),
            $entity->getDataI18n('size')
        );
    }
}
