<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-23
 ******************************************************************************/

namespace Fastmag\Sync\Model\Jobqueue;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterfaceFactory;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagSearchResultsInterfaceFactory;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface;
use Fastmag\Sync\Model\Jobqueue as AbstractJob;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToFastmag as ResourceModel;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToFastmag\CollectionFactory;
use Magento\Framework\Api\DataObjectHelper;
use Magento\Framework\Api\SearchCriteria\CollectionProcessor\FilterProcessor;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\DB\Select;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Reflection\DataObjectProcessor;

/**
 * Class ToFastmagRepository
 *
 * Repository of jobs used for Magento to Fastmag synchronization
 */
class ToFastmagRepository extends StandardRepository implements ToFastmagRepositoryInterface
{
    /** @var ResourceModel $resource */
    protected $resource;

    /** @var ToFastmagFactory $jobFactory */
    protected $jobFactory;

    /** @var CollectionFactory $collectionFactory */
    protected $collectionFactory;

    /** @var ToFastmagSearchResultsInterfaceFactory $searchResultsFactory */
    protected $searchResultsFactory;

    /** @var DataObjectHelper $dataObjectHelper */
    protected $dataObjectHelper;

    /** @var DataObjectProcessor $dataObjectProcessor */
    protected $dataObjectProcessor;

    /** @var ToFastmagInterfaceFactory $jobInterfaceFactory */
    protected $jobInterfaceFactory;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var SortOrderBuilder $sortOrderBuilder */
    protected $sortOrderBuilder;

    /** @var CollectionProcessorInterface $collectionProcessor */
    private $collectionProcessor;

    /**
     * ToFastmagRepository constructor
     *
     * @param ResourceModel                          $resource
     * @param ToFastmagFactory                       $jobFactory
     * @param ToFastmagInterfaceFactory              $jobInterfaceFactory
     * @param CollectionFactory                      $collectionFactory
     * @param ToFastmagSearchResultsInterfaceFactory $searchResultsFactory
     * @param DataObjectHelper                       $dataObjectHelper
     * @param DataObjectProcessor                    $dataObjectProcessor
     * @param SearchCriteriaBuilder                  $searchCriteriaBuilder
     * @param SortOrderBuilder                       $sortOrderBuilder
     * @param CollectionProcessorInterface|null      $collectionProcessor
     */
    public function __construct(
        ResourceModel $resource,
        ToFastmagFactory $jobFactory,
        ToFastmagInterfaceFactory $jobInterfaceFactory,
        CollectionFactory $collectionFactory,
        ToFastmagSearchResultsInterfaceFactory $searchResultsFactory,
        DataObjectHelper $dataObjectHelper,
        DataObjectProcessor $dataObjectProcessor,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SortOrderBuilder $sortOrderBuilder,
        CollectionProcessorInterface $collectionProcessor = null
    ) {
        $this->resource = $resource;
        $this->jobFactory = $jobFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->dataObjectHelper = $dataObjectHelper;
        $this->jobInterfaceFactory = $jobInterfaceFactory;
        $this->dataObjectProcessor = $dataObjectProcessor;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sortOrderBuilder = $sortOrderBuilder;
        $this->collectionProcessor = $collectionProcessor ?: $this->getCollectionProcessor();
    }

    /**
     * @inheritDoc
     */
    public function save(ToFastmagInterface $job)
    {
        try {
            $oldJob = $this->checkExistingJobs($job);
            if ($oldJob !== false) {
                $oldJob->setProcessedAt(null)
                    ->setRetryCount(0)
                    ->setStatus(AbstractJob::STATUS_OK)
                    ->setMessage(null)
                    ->setTrace(null);

                $job = $oldJob;
            }

            $this->resource->save($job);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $job;
    }

    /**
     * @inheritDoc
     */
    public function getById($jobId)
    {
        $job = $this->jobFactory->create();
        $this->resource->load($job, $jobId);
        if (!$job->getId()) {
            throw new NoSuchEntityException(__('The job with the "%1" ID doesn\'t exist.', $jobId));
        }

        return $job;
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $criteria)
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($criteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function getListByCode(SearchCriteriaInterface $criteria)
    {
        $searchResultsArray = [];

        $prioritiesCollection = $this->collectionFactory->create();

        $this->collectionProcessor->process($criteria, $prioritiesCollection);

        $prioritiesCollection->getSelect()->reset(Select::COLUMNS)
            ->columns(ToFastmagInterface::JOB_CODE)
            ->distinct();

        /** @var ToFastmagInterface $row */
        foreach ($prioritiesCollection as $row) {
            $collection = $this->collectionFactory->create();

            $this->searchCriteriaBuilder->setFilterGroups((array)$criteria->getFilterGroups());
            $this->searchCriteriaBuilder->addFilter(ToFastmagInterface::JOB_CODE, $row->getJobCode(), 'eq');

            $this->searchCriteriaBuilder->setSortOrders((array)$criteria->getSortOrders());
            $this->searchCriteriaBuilder->setCurrentPage($criteria->getCurrentPage());
            $this->searchCriteriaBuilder->setPageSize($criteria->getPageSize());

            $rowCriteria = $this->searchCriteriaBuilder->create();

            $this->collectionProcessor->process($rowCriteria, $collection);

            $searchResultsArray[$row->getJobCode()] = $collection;
        }

        return $searchResultsArray;
    }

    /**
     * @inheritDoc
     */
    public function delete(ToFastmagInterface $job)
    {
        try {
            $this->resource->delete($job);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($jobId)
    {
        return $this->delete($this->getById($jobId));
    }

    /**
     * Validate job
     *
     * @param ToFastmagInterface $job
     *
     * @return ToFastmagInterface
     */
    public function reset($job)
    {
        $job->setStatus(AbstractJob::STATUS_OK)
            ->setProcessedAt(date('Y-m-d H:i:s'));

        return $job;
    }

    /**
     * Check if a job already exists for the job code and content ID and returns it
     * If there is more than one, a security has been added to delete the extra jobs
     *
     * @param ToFastmagInterface $job
     *
     * @return ToFastmagInterface|bool
     *
     * @throws CouldNotDeleteException
     * @throws NoSuchEntityException
     */
    protected function checkExistingJobs($job)
    {
        $result = false;

        $sortOrder = $this->sortOrderBuilder->setField(ToFastmagInterface::CREATED_AT)
            ->setAscendingDirection()
            ->create();

        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(ToFastmagInterface::CONTENT_ID, $job->getContentId())
            ->addFilter(ToFastmagInterface::JOB_CODE, $job->getJobCode())
            ->addFilter(ToFastmagInterface::JOB_ID, $job->getId(), 'neq')
            ->addSortOrder($sortOrder)
            ->create();

        $jobsList = $this->getList($searchCriteria);

        if ($jobsList->getTotalCount() > 1) {
            $result = $this->deleteExtraJobs($jobsList->getItems());
        } elseif ($jobsList->getTotalCount() === 1) {
            $jobsListAsArray = $jobsList->getItems();
            $result = reset($jobsListAsArray);
        }

        return $result;
    }

    /**
     * Delete extra jobs if there is more than one job for a specific content ID
     * Returns the first one
     *
     * @param ToFastmagInterface[] $jobsList
     *
     * @return ToFastmagInterface
     *
     * @throws CouldNotDeleteException
     * @throws NoSuchEntityException
     */
    protected function deleteExtraJobs($jobsList)
    {
        $jobsListCount = count($jobsList);

        if ($jobsListCount > 1) {
            for ($i = 1; $i < $jobsListCount; $i++) {
                $oldJob = $jobsList[$i];
                $this->deleteById($oldJob->getId());
            }
        }

        return $jobsList[0];
    }

    /**
     * Retrieve collection processor
     *
     * @deprecated
     *
     * @return CollectionProcessorInterface
     */
    private function getCollectionProcessor()
    {
        if (!$this->collectionProcessor) {
            $this->collectionProcessor = ObjectManager::getInstance()->get(FilterProcessor::class);
        }

        return $this->collectionProcessor;
    }
}
