<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-23
 ******************************************************************************/

namespace Fastmag\Sync\Model\Jobqueue;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterfaceFactory;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoSearchResultsInterfaceFactory;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToMagento as ResourceModel;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToMagento\CollectionFactory;
use Magento\Framework\Api\DataObjectHelper;
use Magento\Framework\Api\FilterBuilder;
use Magento\Framework\Api\SearchCriteria\CollectionProcessor\FilterProcessor;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\DB\Select;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Reflection\DataObjectProcessor;

/**
 * Class ToMagentoRepository
 *
 * Repository of jobs used for Fastmag to Magento synchronization
 */
class ToMagentoRepository extends StandardRepository implements ToMagentoRepositoryInterface
{
    /** @var ResourceModel $resource */
    protected $resource;

    /** @var ToFastmagFactory $jobFactory */
    protected $jobFactory;

    /** @var CollectionFactory $collectionFactory */
    protected $collectionFactory;

    /** @var ToMagentoSearchResultsInterfaceFactory $searchResultsFactory */
    protected $searchResultsFactory;

    /** @var DataObjectHelper $dataObjectHelper */
    protected $dataObjectHelper;

    /** @var DataObjectProcessor $dataObjectProcessor */
    protected $dataObjectProcessor;

    /** @var ToMagentoInterfaceFactory $jobInterfaceFactory */
    protected $jobInterfaceFactory;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var FilterBuilder $filterBuilder */
    protected $filterBuilder;

    /** @var CollectionProcessorInterface */
    private $collectionProcessor;

    /**
     * ToMagentoRepository constructor
     *
     * @param ResourceModel                          $resource
     * @param ToMagentoFactory                       $jobFactory
     * @param ToMagentoInterfaceFactory              $jobInterfaceFactory
     * @param CollectionFactory                      $collectionFactory
     * @param ToMagentoSearchResultsInterfaceFactory $searchResultsFactory
     * @param DataObjectHelper                       $dataObjectHelper
     * @param DataObjectProcessor                    $dataObjectProcessor
     * @param SearchCriteriaBuilder                  $searchCriteriaBuilder
     * @param FilterBuilder                          $filterBuilder
     * @param CollectionProcessorInterface|null      $collectionProcessor
     */
    public function __construct(
        ResourceModel $resource,
        ToMagentoFactory $jobFactory,
        ToMagentoInterfaceFactory $jobInterfaceFactory,
        CollectionFactory $collectionFactory,
        ToMagentoSearchResultsInterfaceFactory $searchResultsFactory,
        DataObjectHelper $dataObjectHelper,
        DataObjectProcessor $dataObjectProcessor,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        FilterBuilder $filterBuilder,
        CollectionProcessorInterface $collectionProcessor = null
    ) {
        $this->resource = $resource;
        $this->jobFactory = $jobFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->dataObjectHelper = $dataObjectHelper;
        $this->jobInterfaceFactory = $jobInterfaceFactory;
        $this->dataObjectProcessor = $dataObjectProcessor;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->filterBuilder = $filterBuilder;
        $this->collectionProcessor = $collectionProcessor ?: $this->getCollectionProcessor();
    }

    /**
     * @inheritDoc
     */
    public function save(ToMagentoInterface $job)
    {
        try {
            $this->resource->save($job);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $job;
    }

    /**
     * @inheritDoc
     */
    public function getById($jobId)
    {
        $job = $this->jobFactory->create();
        $this->resource->load($job, $jobId);
        if (!$job->getId()) {
            throw new NoSuchEntityException(__('The job with the "%1" ID doesn\'t exist.', $jobId));
        }

        return $job;
    }

    /**
     * @inheritDoc
     */
    public function getByUniqueKey($contentId, $jobCode, $priority)
    {
        $job = $this->jobFactory->create();
        $this->resource->loadByUniqueKey($job, $contentId, $jobCode, $priority);
        if (!$job->getId()) {
            throw new NoSuchEntityException(
                __('The job with the following data doesn\'t exist : "%1", "%2", "%3"', $contentId, $jobCode, $priority)
            );
        }

        return $job;
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $criteria)
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($criteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function getListByPriorityAndCode(SearchCriteriaInterface $criteria)
    {
        $searchResultsArray = [];

        // Get priorities and job_code available
        $prioritiesCollection = $this->collectionFactory->create();

        $this->collectionProcessor->process($criteria, $prioritiesCollection);

        $prioritiesCollection->getSelect()->reset(Select::COLUMNS)
            ->columns([ToMagentoInterface::PRIORITY, ToMagentoInterface::JOB_CODE])
            ->distinct()
            ->order(ToMagentoInterface::PRIORITY . ' ' . Select::SQL_DESC);

        /** @var ToMagentoInterface $row */
        foreach ($prioritiesCollection as $row) {
            $collection = $this->collectionFactory->create();

            $this->searchCriteriaBuilder->setFilterGroups((array)$criteria->getFilterGroups());
            $this->searchCriteriaBuilder->addFilter(ToMagentoInterface::PRIORITY, $row->getPriority(), 'eq')
                ->addFilter(ToMagentoInterface::JOB_CODE, $row->getJobCode(), 'eq');

            $this->searchCriteriaBuilder->setSortOrders((array)$criteria->getSortOrders());
            $this->searchCriteriaBuilder->setCurrentPage($criteria->getCurrentPage());
            $this->searchCriteriaBuilder->setPageSize($criteria->getPageSize());

            $rowCriteria = $this->searchCriteriaBuilder->create();

            $this->collectionProcessor->process($rowCriteria, $collection);

            $searchResultsArray[$row->getPriority()][$row->getJobCode()] = $collection;
        }

        return $searchResultsArray;
    }

    /**
     * @inheritDoc
     */
    public function delete(ToMagentoInterface $job)
    {
        try {
            $this->resource->delete($job);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($jobId)
    {
        return $this->delete($this->getById($jobId));
    }

    /**
     * Retrieve collection processor
     *
     * @deprecated
     *
     * @return CollectionProcessorInterface
     */
    private function getCollectionProcessor()
    {
        if (!$this->collectionProcessor) {
            $this->collectionProcessor = ObjectManager::getInstance()->get(FilterProcessor::class);
        }

        return $this->collectionProcessor;
    }
}
