<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-23
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker;

use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Model\Jobqueue;

/**
 * Trait Integration
 *
 * Trait for all integration workers (to Fastmag and to Magento)
 */
trait Integration
{
    use Standard;

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->notice($exception->getMessage());

            return;
        }

        foreach ($this->getJobs()->getItems() as $job) {
            if (!$job->isInError()) {
                try {
                    $this->processJob($job);
                    $this->validateJob($job);
                } catch (JobException $exception) {
                    $this->invalidateJob($job, $exception);
                }

                $this->saveJob($job);
            }
        }
    }

    /**
     * Returns the activation of the synchronization of the current entity
     *
     * @return bool
     */
    abstract public function isEnabled();

    /**
     * Returns the activation of the synchronization of the current entity
     *
     * @return bool
     */
    abstract protected function processJob($job);

    /**
     * Check initial conditions to run the worker
     *
     * @throws ProcessException
     */
    protected function initiate()
    {
        if (!$this->isEnabled()) {
            throw new ProcessException(__('Worker "%1" was called, even though it is disabled', $this->code));
        } elseif ($this->getJobs()->count() <= 0) {
            throw new ProcessException(__('Worker "%1" was called, but without jobs to integrate', $this->code));
        }
    }

    /**
     * Validate job
     *
     * @param Jobqueue $job
     *
     * @return Standard
     */
    public function validateJob($job)
    {
        $this->jobRepository->validate($job);

        $this->log($job, 'Job validated');

        return $this;
    }
}
