<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-13
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\RemoteSync\Sync;

use Fastmag\Sync\Api\Jobqueue\ToMagentoScheduledRepositoryInterface;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Jobqueue\ToMagentoScheduledFactory;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Worker\RemoteSync\Sync;
use Magento\Framework\App\ResourceConnection;
use Magento\Store\Api\WebsiteRepositoryInterface as WebsiteRepository;
use Magento\Store\Model\ScopeInterface;

/**
 * Class Scheduled
 *
 * Sync worker class used for sync all jobs which have to be processed in a scheduled time
 */
class Scheduled extends Sync
{
    /** @inheritDoc */
    protected $code = 'remotesync_sync_scheduled';

    /** @var ToMagentoScheduledFactory $jobFactory */
    protected $jobFactory;

    /** @var ToMagentoScheduledRepositoryInterface $jobRepository */
    protected $jobRepository;

    /** @var Config $config */
    protected $config;

    /** @var WebsiteRepository $websiteRepository */
    protected $websiteRepository;

    /**
     * Instant constructor
     *
     * @param Logger                                $logger
     * @param ResourceConnection                    $resourceConnection
     * @param FastmagSql                            $fastmagSql
     * @param ToMagentoScheduledFactory             $jobFactory
     * @param ToMagentoScheduledRepositoryInterface $jobRepository
     * @param Config                                $config
     * @param WebsiteRepository                     $websiteRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        FastmagSql $fastmagSql,
        ToMagentoScheduledFactory $jobFactory,
        ToMagentoScheduledRepositoryInterface $jobRepository,
        Config $config,
        WebsiteRepository $websiteRepository
    ) {
        parent::__construct($logger, $resourceConnection, $fastmagSql);

        $this->jobFactory = $jobFactory;
        $this->jobRepository = $jobRepository;
        $this->config = $config;
        $this->websiteRepository = $websiteRepository;
    }

    /**
     * @inheritDoc
     */
    protected function saveJob($event)
    {
        $createdAtDate = $this->getUtcDateTimeFromFastmag($event['date_maj']);
        $scheduledAtDate = $this->getUtcDateTimeFromFastmag($event['date_a_traiter']);

        $job = $this->jobFactory->create();
        $job->setContentId($event['identifiant'])
            ->setJobCode($this->generateJobCode($event['job']))
            ->setFastmagShop($event['code_mag'])
            ->setContent($event['commentaire'])
            ->setCreatedAt($event['date_maj'])
            ->setScheduledAt($scheduledAtDate);
        $this->jobRepository->save($job);

        $newLastExecution = $createdAtDate;

        $this->logger->debug('Last Id : ' . $event['identifiant'] . ' - Last date : ' . $newLastExecution);
    }

    /**
     * @inheritDoc
     */
    protected function getSqlCondition()
    {
        $shopsList = $this->getShopsList();

        $shopsCondition = '';
        if ($shopsList !== '') {
            $shopsCondition = ' AND code_mag IN (' . $this->getFastmagSqlConnection()->escape($shopsList, true) . ')';
        }

        return '(date_a_traiter IS NOT NULL AND date_a_traiter > NOW())' . $shopsCondition;
    }

    /**
     * Get the shops list from the config
     *
     * @return array
     */
    protected function getShopsList()
    {
        $result = [];

        $result[] = $this->config->getValue(Config::XML_PATH_ORDER_WORKFLOW_SHOP);

        foreach ($this->websiteRepository->getList() as $website) {
            $result[] = $this->config->getValue(
                Config::XML_PATH_ORDER_WORKFLOW_SHOP,
                ScopeInterface::SCOPE_WEBSITES,
                $website->getId()
            );
        }

        return array_unique($result);
    }
}
