<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-24
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Order;

use Fastmag\Sync\Api\CustomerRepositoryInterface as SyncedCustomerRepository;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\CustomerFactory as CustomerEntityFactory;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer as AbstractCustomer;
use Magento\Customer\Api\CustomerRepositoryInterface as CustomerRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Newsletter\Model\Subscriber;
use Magento\Sales\Api\OrderRepositoryInterface as OrderRepository;

class Customer extends AbstractCustomer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_order_customer';

    /** @var OrderRepository $orderRepository */
    protected $orderRepository;

    /** @var Subscriber $subscriber */
    protected $subscriber;

    /** @var array $ordersToCustomers */
    protected $ordersToCustomers = [];

    /**
     * Customer constructor.
     *
     * @param Logger                   $logger
     * @param JobRepository            $jobRepository
     * @param Config                   $config
     * @param Json                     $jsonSerializer
     * @param SyncedCustomerRepository $syncedCustomerRepository
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param CustomerRepository       $customerRepository
     * @param CustomerEntityFactory    $customerEntityFactory
     * @param OrderRepository          $orderRepository
     * @param Subscriber               $subscriber
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        SyncedCustomerRepository $syncedCustomerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        CustomerRepository $customerRepository,
        CustomerEntityFactory $customerEntityFactory,
        OrderRepository $orderRepository,
        Subscriber $subscriber
    ) {
        parent::__construct(
            $logger,
            $jobRepository,
            $config,
            $jsonSerializer,
            $syncedCustomerRepository,
            $searchCriteriaBuilder,
            $customerRepository,
            $customerEntityFactory
        );

        $this->orderRepository = $orderRepository;
        $this->subscriber = $subscriber;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        $entities = false;

        try {
            $entities = $this->getDataFromMagento();
        } catch (ProcessException $exception) {
            foreach ($this->getJobs()->getItems() as $job) {
                $this->invalidateJob($job, $exception);
            }
        }

        if (is_array($entities)) {
            foreach ($this->ordersToCustomers as $contentId => $customerId) {
                $job = $this->getJob($contentId);

                if ($job !== null && !$job->isInError()) {
                    /** @var OrderEntity $orderEntity */
                    $orderEntity = $job->getEntity();
                    $orderEntity->setCustomer($entities[$customerId]);

                    $this->hydrateJob($job, $orderEntity);
                    $this->log($job, 'Order\'s customer hydrated');
                }
            }
        }

        foreach ($this->getJobs()->getItems() as $job) {
            $this->saveJob($job);
        }
    }

    /**
     * @inheritDoc
     *
     * @return CustomerEntity[]
     *
     * @throws ProcessException
     */
    protected function getDataFromMagento()
    {
        $customerIds = $this->getCustomerIds();

        foreach ($customerIds as $customerId) {
            $customerEntity = $this->customerEntityFactory->create();
            $customerEntity->setMagentoId($customerId);
            $this->entities[$customerId] = $customerEntity;
        }

        $this->getSyncedCustomers($customerIds);
        $this->getMagentoCustomersData($customerIds);

        return $this->entities;
    }

    /**
     * Retrieve the customer IDs for the orders' jobs
     *
     * @return array
     */
    protected function getCustomerIds()
    {
        $customerIds = [];

        foreach ($this->getJobs()->getItems() as $job) {
            /** @var OrderEntity $orderEntity */
            $orderEntity = $job->getEntity();

            $transactionType = $orderEntity->getTransactionType();
            $orderId = $orderEntity->getMagentoId();

            if ($orderId !== null && ($transactionType === null || in_array(
                $transactionType,
                [
                    OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE,
                    OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION,
                    OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_ORDER
                ],
                true
            ))) {
                $customerId = $orderEntity->getCustomerId();
                if ($customerId === null) {
                    continue;
                }

                $this->ordersToCustomers[$job->getContentId()] = $customerId;
                $customerIds[] = $customerId;
            }
        }

        return $customerIds;
    }
}
