<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-23
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration;

use Exception;
use Fastmag\Sync\Api\Data\CustomerInterfaceFactory as SyncedCustomerFactory;
use Fastmag\Sync\Api\Data\CustomerInterface as SyncedCustomer;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\CustomerRepositoryInterface as SyncedCustomerRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Api;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration;
use Fastmag\Sync\Process\Worker\FastmagSql as FastmagSqlTrait;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Customer
 *
 * Abstract class for Customer related Integration workers
 */
abstract class Customer extends Integration
{
    use FastmagSqlTrait;

    /** @var Config $config */
    protected $config;

    /** @var SyncedCustomerFactory $syncedCustomerFactory */
    protected $syncedCustomerFactory;

    /** @var SyncedCustomerRepository $syncedCustomerRepository */
    protected $syncedCustomerRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /**
     * Customer constructor.
     *
     * @param Logger                   $logger
     * @param JobRepository            $jobRepository
     * @param Api                      $api
     * @param FastmagSql               $fastmagSql
     * @param Config                   $config
     * @param SyncedCustomerFactory    $syncedCustomerFactory
     * @param SyncedCustomerRepository $syncedCustomerRepository
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Api $api,
        FastmagSql $fastmagSql,
        Config $config,
        SyncedCustomerFactory $syncedCustomerFactory,
        SyncedCustomerRepository $syncedCustomerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder
    ) {
        parent::__construct($logger, $jobRepository, $api);

        $this->fastmagSql = $fastmagSql;
        $this->config = $config;
        $this->syncedCustomerFactory = $syncedCustomerFactory;
        $this->syncedCustomerRepository = $syncedCustomerRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return true;
    }

    /**
     * Save sync customer
     *
     * @param int  $magentoId
     * @param int  $fastmagId
     * @param bool $isDeleted
     *
     * @throws JobException
     */
    protected function saveSyncCustomer($magentoId, $fastmagId, $isDeleted = false)
    {
        $searchCriteria = $this->searchCriteriaBuilder->addFilter(SyncedCustomer::MAGENTO_CUSTOMER_ID, $magentoId)
            ->create();

        $alreadySyncedCustomers = $this->syncedCustomerRepository->getList($searchCriteria);

        if ($alreadySyncedCustomers->getTotalCount() > 1) {
            throw new JobException(
                __('Customer #%1 has already been synced with Fastmag and have multiple Fastmag ID', $magentoId)
            );
        }

        if ($alreadySyncedCustomers->getTotalCount() === 1) {
            try {
                $syncCustomer = $this->syncedCustomerRepository->getByMagentoId($magentoId);
                $syncCustomer->setFastmagCustomerId($fastmagId)
                    ->setIsDeleted($isDeleted);
            } catch (Exception $exception) {
                throw new JobException(
                    __('Unable to get customer #%1 Fastmag ID on synced customers table', $magentoId)
                );
            }
        } else {
            $syncCustomer = $this->syncedCustomerFactory->create();
            $syncCustomer->setMagentoCustomerId($magentoId)
                ->setFastmagCustomerId($fastmagId)
                ->setIsDeleted($isDeleted);
        }

        try {
            $this->syncedCustomerRepository->save($syncCustomer);
        } catch (CouldNotSaveException $exception) {
            throw new JobException(__('Unable to save customer\'s Fastmag ID: %1', $exception->getMessage()));
        }
    }

    /**
     * Check if customer email exists in Fastmag and returns Fastmag ID
     *
     * @param string $email
     *
     * @return int|null
     *
     * @throws JobException
     */
    protected function getFastmagIdByEmail($email)
    {
        $result = null;

        try {
            $sql = 'SELECT c.Client AS fastmag_id
                FROM (
                    SELECT MAX(TotalCA) AS MaxTotalCA, Email
                    FROM clients
                    WHERE Email = ' . $this->getFastmagSqlConnection()->escape($email) . '
                    GROUP BY Email
                ) AS ce INNER JOIN clients AS c
                    ON c.Email = ce.Email AND c.TotalCA = ce.MaxTotalCA';

            $row = $this->getFastmagSqlConnection()->get($sql);

            if (count($row) > 0) {
                $result = (int)$row[0]['fastmag_id'];
            }

            return $result;
        } catch (Exception $exception) {
            throw new JobException(__(
                'Error when trying to get customer ID by email. Message: %1. Customer email: %2',
                $exception->getMessage(),
                $email
            ));
        }
    }
}
