<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-23
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration;

/**
 * Class Order
 *
 * Hydration class used for inserting and updating orders from Fastmag to Magento
 */
class Order extends Hydration
{
    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $results = $this->getDataFromFastmag();

            foreach ($results as $row) {
                $job = $this->getJob($row['order_id']);

                if ($job !== null) {
                    $this->hydrateJob($job, $row);
                }
            }

            foreach ($this->getJobs()->getItems() as $job) {
                $this->saveJob($job);
            }
        } catch (ProcessException $exception) {
            foreach ($this->getJobs()->getItems() as $job) {
                $this->invalidateJob($job, $exception);
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    protected function getDataFromFastmag()
    {
        $ordersIds = $this->getJobs()->getColumnValues(Job::CONTENT_ID);

        try {
            $sql = 'SELECT Vente AS order_id, Client AS customer_id, Date AS creation_date, Heure AS creation_time,
                Total AS total, CodeMag AS shop_code, VShopStatus AS status, ListeColis AS tracking_number,
                (
                    SELECT TRIM(valeurstring)
                    FROM complementsr AS csr
                    WHERE csr.Champ = \'NOM_WEB\' AND csr.Nature = \'MAGASINS\' AND csr.Clef = v.CodeMag
                ) AS alt_shop_code,
                (SELECT r.Libelle FROM reglements AS r WHERE r.Vente = v.Vente LIMIT 1) AS payment_method
                FROM ventes AS v
                WHERE v.Nature = \'VENTE\'
                    AND v.Client > 0
                    AND v.CodeMag NOT IN (\'CENTRAL\')
                    AND (v.InfosComp IS NULL OR v.InfosComp NOT LIKE \'Magento%\')
                    AND v.Vente IN (' . $this->getFastmagSqlConnection()->escape($ordersIds) . ')';

            return $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new ProcessException(__(
                'Error when hydrating orders. Message: %1. Orders IDs: %2',
                $exception->getMessage(),
                implode(', ', $ordersIds)
            ));
        }
    }

    /**
     * Returns the job for the current order ID hydrated
     *
     * @param int $orderFastmagId
     *
     * @return Job
     */
    protected function getJob($orderFastmagId)
    {
        return $this->getJobs()->getItemByColumnValue(Job::CONTENT_ID, $orderFastmagId);
    }
}
