<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-24
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product;

use Exception;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product;
use Monolog\Logger as Monolog;

/**
 * Class Price
 *
 * Hydration class used for inserting or updating products prices from Fastmag to Magento
 */
class Price extends Product
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_product_price';

    /** @inheritDoc */
    protected $syncConfigFields = [
        Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS,
        Config::XML_PATH_PRODUCT_PRICE_FASTMAG_RATE,
        Config::XML_PATH_INVENTORY_FASTMAG_STOCK_REFERENCE_STOCK,
        Config::XML_PATH_INVENTORY_FASTMAG_STOCK_ALTERNATIVE_STOCKS,
        Config::XML_PATH_ORDER_WORKFLOW_SHOP
    ];

    /**
     * @inheritDoc
     */
    public function run()
    {
        $this->getSyncConfig();

        foreach ($this->getJobs()->getItems() as $job) {
            $this->currentJob = $job;

            $fastmagRef = $this->getCurrentFastmagRef();

            if ($this->getMagentoProductId($fastmagRef) !== false) {
                try {
                    $entity = $this->getDataFromFastmag();

                    $this->hydrateJob($job, $entity);
                } catch (JobException $exception) {
                    $this->invalidateJob($job, $exception);
                }

                $this->saveJob($job);
            } else {
                $this->log(
                    $job,
                    'Can not change prices level of the product with Fastmag ref "' . $fastmagRef . '"'
                    . ', as it does not exist in Magento',
                    Monolog::NOTICE
                );
            }
        }
    }

    /**
     * @inheritDoc
     */
    protected function initiateCurrentEntity($fastmagRef)
    {
        $this->currentEntity = $this->productEntityFactory->create();
        $this->currentEntity->setRef($fastmagRef)
            ->setMagentoId($this->getMagentoProductId($fastmagRef));
    }

    /**
     * @inheritDoc
     */
    protected function getDataByStore()
    {
        foreach (array_keys($this->syncConfig['stores']) as $storeId) {
            $this->getChildrenData($storeId);
        }
    }

    /**
     * @inheritDoc
     */
    protected function getParentData()
    {
        try {
            $sql = 'SELECT pf.VisibleWeb AS visible_web,
                    pf.Actif AS active,
                    COUNT(s.ID) AS stock_level,
                    f.Famille AS vat_family,
                    f.TauxTVA AS vat_rate
                FROM produitsfiches AS pf LEFT JOIN stock AS s on pf.BarCode = s.BarCode
                    LEFT JOIN familles AS f ON pf.Famille = f.Famille
                WHERE pf.BarCode =
                    ' . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
                    AND s.AR = 1
                GROUP BY pf.BarCode';

            $rows = $this->getFastmagSqlConnection()->get($sql);

            if (count($rows) === 0) {
                $sql = 'SELECT pf.VisibleWeb AS visible_web,
                    pf.Actif AS active,
                    0 AS stock_level,
                    f.Famille AS vat_family,
                    f.TauxTVA AS vat_rate
                FROM produitsfiches AS pf LEFT JOIN stock AS s on pf.BarCode = s.BarCode
                    LEFT JOIN familles AS f ON pf.Famille = f.Famille
                WHERE pf.BarCode = '
                    . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
                GROUP BY pf.BarCode';

                $rows = $this->getFastmagSqlConnection()->get($sql);
            }
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        $row = reset($rows);

        $this->currentEntity->addData($row);
    }

    /**
     * @inheritDoc
     */
    protected function getChildrenDataSql($storeId = null)
    {
        $fastmagShop = $this->getSyncConfigValueFromArray(Config::XML_PATH_ORDER_WORKFLOW_SHOP, $storeId);
        $rateCode = $this->getSyncConfigValueFromArray(Config::XML_PATH_PRODUCT_PRICE_FASTMAG_RATE, $storeId);

        $standardPriceSubquery = $this->getStandardPriceSubquery($fastmagShop, $rateCode);
        $combinationPriceSubquery = $this->getCombinationPriceSubquery($rateCode);
        $ratePriceSubquery = $this->getRatePriceSubquery($rateCode);
        $salesPriceSubquery = $this->getSalesPriceSubquery($rateCode);
        $salesPriceChildrenSubquery = $this->getSalesPriceChildrenSubquery($fastmagShop);
        $buyingPriceSubquery = $this->getBuyingPriceSubquery();

        return 'SELECT
                (
                    SELECT 1
                    FROM stock AS s2
                    WHERE s2.BarCode = stock.BarCode AND s2.Taille = stock.Taille AND s2.Couleur = stock.Couleur
                        ' . $this->getStocksCodesCondition($storeId) . '
                    LIMIT 1
                ) AS active_stock,
                pf.PrixVente AS indicative_price,
                (' . $standardPriceSubquery . ') AS standard_price,
                (' . $combinationPriceSubquery . ') AS combination_price,
                (' . $ratePriceSubquery . ') AS rate_price,
                (' . $salesPriceSubquery . ') AS sales_price,
                (' . $salesPriceChildrenSubquery . ') AS sales_price_children,
                (' . $buyingPriceSubquery . ') AS buying_price,
                stock.Taille AS size,
                stock.Couleur AS color
            FROM produitsfiches AS pf INNER JOIN stock ON pf.BarCode = stock.BarCode
            WHERE stock.AR = 1
                AND stock.BarCode =
                ' . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
            GROUP BY stock.Barcode, stock.Taille, stock.Couleur
            ' . $this->getOnlyDefinedStocksCondition($storeId) . '
            ORDER BY stock.Couleur, stock.Taille';
    }

    /**
     * @inheritDoc
     */
    protected function getOnlyDefinedStocksCondition($storeId = null)
    {
        $result = '';

        if ($this->getSyncConfigValueFromArray(Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS, $storeId)) {
            $result = 'HAVING active_stock = 1 ';
        }

        return $result;
    }
}
