<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-25
 ******************************************************************************/

namespace Fastmag\Sync\Process;

use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\Collection;
use Fastmag\Sync\Process\Worker\Standard;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration as ToFastmagHydration;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration as ToMagentoHydration;

/**
 * Class Manager
 *
 * Parent class of the process manager (remote sync, to magento and to fastmag)
 */
class Manager
{
    /** @var Logger $logger */
    protected $logger;

    /** @var Config $config */
    protected $config;

    /** @var WorkerFactory $workerFactory */
    protected $workerFactory;

    /**
     * Job queue manager constructor.
     *
     * @param Logger        $logger
     * @param Config        $config
     * @param WorkerFactory $workerFactory
     */
    public function __construct(
        Logger $logger,
        Config $config,
        WorkerFactory $workerFactory
    ) {
        $this->logger = $logger;
        $this->config = $config;
        $this->workerFactory = $workerFactory;
    }

    /**
     * Check required config fields to run the synchronization
     *
     * @return bool
     *
     * @throws ProcessException
     */
    protected function checkPrerequisites()
    {
        $result = true;

        if (!$this->isSyncEnabled()) {
            throw new ProcessException(__('Synchronization is currently disabled. Remote sync can not be done.'));
        }

        if ($this->getSyncLimit() <= 0) {
            throw new ProcessException(__('Process queue limit size not set in config.'));
        }

        return $result;
    }

    /**
     * Check if global synchronization is enabled
     *
     * @return bool
     */
    protected function isSyncEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_JOBQUEUE_SYNC_ENABLED);
    }

    /**
     * Get sync limit
     *
     * @return int
     */
    protected function getSyncLimit()
    {
        return (int)$this->config->getValue(Config::XML_PATH_JOBQUEUE_ADVANCED_JOBS_LIMIT);
    }

    /**
     * Run worker and subordinate workers if the worker has any
     *
     * @param Standard   $worker
     * @param Collection $jobsCollection
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function runWorker($worker, $jobsCollection)
    {
        if ($worker->hasSubordinateWorkersBefore()) {
            foreach ($worker->getSubordinateWorkersBefore() as $subWorker) {
                $subWorker = $this->workerFactory->create($subWorker);
                /** @var Standard $subWorker */
                $this->runWorker($subWorker, $jobsCollection);
            }
        }

        $worker->setJobs($jobsCollection)->run();

        if ($worker->hasSubordinateWorkersAfter()) {
            foreach ($worker->getSubordinateWorkersAfter() as $subWorker) {
                $subWorker = $this->workerFactory->create($subWorker);
                /** @var Standard $subWorker */
                $this->runWorker($subWorker, $jobsCollection);
            }
        }
    }

    /**
     * Run hydration workers to hydrate current jobs and add entities created to the jobs collection
     *
     * @param string     $jobCode
     * @param Collection $jobsCollection
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function hydrateJobs($jobCode, $jobsCollection)
    {
        if ($jobsCollection->count() > 0) {
            $hydrationJobCode = $this->getHydrationJobCode($jobCode);

            if ($hydrationJobCode !== '') {
                /** @var ToFastmagHydration|ToMagentoHydration $worker */
                $worker = $this->workerFactory->create($hydrationJobCode);

                /** @throws ProcessException */
                $this->runWorker($worker, $jobsCollection);
            }
        }
    }
}
