<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-23
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Helper\Text;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Process\Entity\ToFastmag\Address as AddressEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\AddressFactory as AddressEntityFactory;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration;
use Magento\Customer\Api\AddressRepositoryInterface as AddressRepository;
use Magento\Customer\Api\CustomerRepositoryInterface as CustomerRepository;
use Magento\Customer\Api\Data\AddressInterface as AddressModel;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Sales\Api\Data\OrderInterface as OrderModel;
use Magento\Sales\Api\Data\OrderAddressInterface as OrderAddressModel;

/**
 * Class Address
 *
 * Hydration class used for inserting or updating customers addresses from Magento to Fastmag
 */
abstract class Address extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_address';

    /** @var CustomerRepository $customerRepository */
    protected $customerRepository;

    /** @var AddressRepository $addressRepository */
    protected $addressRepository;

    /** @var AddressEntityFactory $addressEntityFactory */
    protected $addressEntityFactory;

    /**
     * Address constructor.
     *
     * @param Logger               $logger
     * @param JobRepository        $jobRepository
     * @param Config               $config
     * @param Json                 $jsonSerializer
     * @param CustomerRepository   $customerRepository
     * @param AddressRepository    $addressRepository
     * @param AddressEntityFactory $addressEntityFactory
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        CustomerRepository $customerRepository,
        AddressRepository $addressRepository,
        AddressEntityFactory $addressEntityFactory
    ) {
        parent::__construct($logger, $jobRepository, $config, $jsonSerializer);

        $this->customerRepository = $customerRepository;
        $this->addressRepository = $addressRepository;
        $this->addressEntityFactory = $addressEntityFactory;
    }

    /**
     * Get customer's last order addresses or default addresses if there is no order
     *
     * @param int        $customerId
     * @param OrderModel $lastOrder
     * @param string     $type
     *
     * @return AddressEntity
     */
    protected function getAddressEntity($customerId, $lastOrder, $type = 'billing')
    {
        $result = null;
        $addressObject = null;

        if ($lastOrder !== false) {
            if ($type === 'billing') {
                $addressObject = $lastOrder->getBillingAddress();
            } else {
                $addressObject = $lastOrder->getShippingAddress();
            }
        } else {
            try {
                $customer = $this->customerRepository->getById($customerId);
            } catch (Exception $exception) {
                $customer = null;
            }

            if ($customer !== null) {
                if ($type === 'billing') {
                    $addressId = $customer->getDefaultBilling();
                } else {
                    $addressId = $customer->getDefaultShipping();
                }

                if ($addressId) {
                    try {
                        $addressObject = $this->addressRepository->getById($addressId);
                    } catch (Exception $exception) {
                        $addressObject = null;
                    }
                }
            }
        }

        if ($addressObject !== null) {
            $result = $this->addressEntityFactory->create();

            $result->setAlias($this->getAddressAlias($addressObject, $lastOrder, $customerId, $type))
                ->setMagentoId($addressObject->getId())
                ->setFirstname($addressObject->getFirstname())
                ->setLastname($addressObject->getLastname())
                ->setCompany($addressObject->getCompany())
                ->setStreet($addressObject->getStreet())
                ->setPostcode($addressObject->getPostcode())
                ->setCity($addressObject->getCity())
                ->setCountryId($addressObject->getCountryId())
                ->setPhoneNumber($addressObject->getTelephone());

            if ($type === 'shipping') {
                $result->setArchived((bool)$lastOrder);
            }
        }

        return $result;
    }

    /**
     * Defines the address alias
     *
     * @param AddressModel|OrderAddressModel $shippingAddress
     * @param OrderModel|null                $lastOrder
     * @param int                            $customerId
     * @param string                         $addressType
     *
     * @return string
     */
    protected function getAddressAlias($shippingAddress, $lastOrder, $customerId, $addressType = 'billing')
    {
        $result = '';

        if ($addressType === 'billing') {
            $result = ($lastOrder !== false ? $lastOrder->getEntityId() . '_INVOICE' : 'INVOICE');
        } elseif ($addressType === 'shipping') {
            $shippingAddressAlias = $shippingAddress->getCustomAttribute(Constants::ATTRIBUTE_ADDRESS_ALIAS_CODE);
            if ($shippingAddressAlias !== null) {
                $shippingAddressAlias = $shippingAddressAlias->getValue();
            }

            if ($shippingAddressAlias === '.' || $shippingAddressAlias === '') {
                $shippingAddressAlias = $customerId;
            }

            if ($lastOrder && $lastOrder->getEntityId()) {
                $shippingAddressAlias = $lastOrder->getEntityId() . '_' . $shippingAddressAlias;
            }

            $result = Text::upper(substr(Text::sanitize(Text::removeAccents($shippingAddressAlias)), 0, 20));
        }

        return $result;
    }

    /**
     * Returns the job for the current customer ID hydrated
     *
     * @param int $customerId
     *
     * @return Job
     */
    protected function getJob($customerId)
    {
        return $this->getJobs()->getItemByColumnValue(Job::CONTENT_ID, $customerId);
    }
}
