<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-25
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Entity\ToMagento\Product as ProductEntity;
use Fastmag\Sync\Process\Entity\ToMagento\ProductFactory as ProductEntityFactory;
use Fastmag\Sync\Process\Entity\ToMagento\Product\VariationFactory as VariationEntityFactory;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration;
use InvalidArgumentException;
use Magento\Catalog\Helper\Data;
use Magento\Catalog\Model\ProductFactory;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\InventoryApi\Api\StockRepositoryInterface as StockRepository;
use Magento\Store\Api\StoreRepositoryInterface as StoreRepository;
use Magento\Store\Model\ScopeInterface;
use Monolog\Logger as Monolog;

/**
 * Class Product
 *
 * Hydration class used for inserting or updating products from Fastmag to Magento
 */
class Product extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_product';

    /** @var ProductFactory $productFactory */
    protected $productFactory;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var StoreRepository $storeRepository */
    protected $storeRepository;

    /** @var ProductEntityFactory $productEntityFactory */
    protected $productEntityFactory;

    /** @var VariationEntityFactory $variationEntityFactory */
    protected $variationEntityFactory;

    /** @var StockRepository $stockRepository */
    protected $stockRepository;

    /** @var Job $currentJob */
    protected $currentJob;

    /** @var ProductEntity $currentEntity */
    protected $currentEntity;

    /** @var array $colorsI18n */
    protected $colorsI18n;

    /** @var array $sizesI18n */
    protected $sizesI18n;

    /** @var array $syncConfig */
    protected $syncConfig;

    /** @var string[] $syncConfigFields */
    protected $syncConfigFields = [
        Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS,
        Config::XML_PATH_PRODUCT_PRICE_FASTMAG_RATE,
        Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_LANGUAGE,
        Config::XML_PATH_INVENTORY_FASTMAG_STOCK_REFERENCE_STOCK,
        Config::XML_PATH_INVENTORY_FASTMAG_STOCK_ALTERNATIVE_STOCKS,
        Config::XML_PATH_ORDER_WORKFLOW_SHOP
    ];

    /**
     * Product constructor
     *
     * @param Logger                    $logger
     * @param JobRepository             $jobRepository
     * @param Config                    $config
     * @param Json                      $jsonSerializer
     * @param FastmagSql                $fastmagSql
     * @param ProductFactory            $productFactory
     * @param SearchCriteriaBuilder     $searchCriteriaBuilder
     * @param StoreRepository           $storeRepository
     * @param ProductEntityFactory      $productEntityFactory
     * @param VariationEntityFactory    $variationEntityFactory
     * @param StockRepository           $stockRepository
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        FastmagSql $fastmagSql,
        ProductFactory $productFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        StoreRepository $storeRepository,
        ProductEntityFactory $productEntityFactory,
        VariationEntityFactory $variationEntityFactory,
        StockRepository $stockRepository
    ) {
        parent::__construct($logger, $jobRepository, $config, $jsonSerializer, $fastmagSql);

        $this->productFactory = $productFactory;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->storeRepository = $storeRepository;
        $this->productEntityFactory = $productEntityFactory;
        $this->variationEntityFactory = $variationEntityFactory;
        $this->stockRepository = $stockRepository;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        $this->getSyncConfig();
        $this->getGlobalData();

        foreach ($this->getJobs()->getItems() as $job) {
            $this->currentJob = $job;

            try {
                $entity = $this->getDataFromFastmag();

                $this->hydrateJob($job, $entity);
            } catch (JobException $exception) {
                $this->invalidateJob($job, $exception);
            }

            $this->saveJob($job);
        }
    }

    /**
     * Hydrate the product of the current job
     *
     * @return ProductEntity
     *
     * @throws JobException
     */
    protected function getDataFromFastmag()
    {
        $fastmagRef = $this->currentJob->getContentId();

        if ($this->isPack($fastmagRef)) {
            throw new JobException(__(
                'Product "%1" is a pack, which can not be imported with Fastmag_Sync module',
                $fastmagRef
            ));
        }

        $this->initiateCurrentEntity($fastmagRef);

        $this->getParentData();

        if ($this->productMustBeSynced()) {
            $this->getChildrenData();

            if (array_key_exists('stores', $this->syncConfig)) {
                $this->getDataByStore();
            }

            if ($this->currentEntity->getVariations() === null) {
                throw new JobException(
                    __('No existing or salable variation on Fastmag for the product "%1"', $fastmagRef)
                );
            }
        } else {
            $this->log(
                $this->currentJob,
                'Product "' . $fastmagRef . '" can\'t be synced, it must be inactive or not in stock: '
                    . $this->jsonSerializer->serialize($this->currentEntity->export()),
                Monolog::NOTICE
            );
        }

        return $this->currentEntity;
    }

    /**
     * Initiate current product entity
     *
     * @param string $fastmagRef
     *
     * @return void
     */
    protected function initiateCurrentEntity($fastmagRef)
    {
        $this->currentEntity = $this->productEntityFactory->create();
        $this->currentEntity->setRef($fastmagRef)
            ->setColorsI18n($this->colorsI18n)
            ->setSizesI18n($this->sizesI18n)
            ->setMagentoId($this->getMagentoProductId($fastmagRef));
    }

    /**
     * Get product data for each store
     *
     * @return void
     *
     * @throws JobException
     */
    protected function getDataByStore()
    {
        foreach (array_keys($this->syncConfig['stores']) as $storeId) {
            $productI18nEntity = $this->productEntityFactory->create();
            $this->currentEntity->addI18n($productI18nEntity, $storeId);

            $this->getFastmagShopTaxRate($storeId);
            $this->getI18nData($storeId);
            $this->getChildrenData($storeId);
        }
    }

    /**
     * Get global data (independent of the jobs)
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getGlobalData()
    {
        $this->getColorsI18n();
        $this->getSizesI18n();
    }

    /**
     * Get colors labels i18n
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getColorsI18n()
    {
        try {
            $sql = 'SELECT Langue AS language, Couleur AS color, Traduction AS translation FROM couleurslangues';

            $this->colorsI18n = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new ProcessException(__($exception->getMessage()));
        }
    }

    /**
     * Get sizes labels i18n
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getSizesI18n()
    {
        try {
            $sql = 'SELECT Langue AS language, Taille AS size, Traduction AS translation FROM tailleslangues';

            $this->sizesI18n = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new ProcessException(__($exception->getMessage()));
        }
    }

    /**
     * Check if product is a pack
     *
     * @param string $fastmagRef
     *
     * @return bool
     */
    protected function isPack($fastmagRef)
    {
        return (strpos($fastmagRef, 'PACK_') !== false);
    }

    /**
     * Get Magento product ID for Fastmag ref given
     *
     * @param string $fastmagRef
     *
     * @return int|false
     */
    protected function getMagentoProductId($fastmagRef)
    {
        $product = $this->productFactory->create();

        return $product->getIdBySku($fastmagRef);
    }

    /**
     * Get product basic data on Fastmag (VisibleWeb, Actif, creation date, inventory data, family)
     *
     * @return void
     *
     * @throws JobException
     */
    protected function getParentData()
    {
        $additionalAttributes = $this->getAdditionalAttributes('pf');

        try {
            $sql = 'SELECT pf.VisibleWeb AS visible_web,
                    pf.Actif AS active,
                    COUNT(s.ID) AS stock_level,
                    GROUP_CONCAT(DISTINCT IF(Stock >= 1, s.CodeMag, NULL)) AS stocks_list,
                    f.Famille AS vat_family,
                    f.TauxTVA AS vat_rate,
                    TRIM(pf.Marque) AS brand,
                    TRIM(pf.Fournisseur) AS supplier,
                    IF(pf.Poids > 0, pf.Poids, 0.1) AS weight,
                    TRIM(pf.Designation) AS designation,
                    TRIM(pf.Designation2) AS designation_bis
                    ' .  $additionalAttributes . '
                FROM produitsfiches AS pf LEFT JOIN stock AS s on pf.BarCode = s.BarCode
                    LEFT JOIN familles AS f ON pf.Famille = f.Famille
                WHERE pf.BarCode = '
                    . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
                    AND s.AR = 1
                GROUP BY pf.BarCode';

            $rows = $this->getFastmagSqlConnection()->get($sql);

            if (count($rows) === 0) {
                $sql = 'SELECT pf.VisibleWeb AS visible_web,
                    pf.Actif AS active,
                    0 AS stock_level,
                    \'\' AS stocks_list,
                    f.Famille AS vat_family,
                    f.TauxTVA AS vat_rate,
                    TRIM(pf.Marque) AS brand,
                    TRIM(pf.Fournisseur) AS supplier,
                    IF(pf.Poids > 0, pf.Poids, 0.1) AS weight,
                    TRIM(pf.Designation) AS designation,
                    TRIM(pf.Designation2) AS designation_bis
                    ' .  $additionalAttributes . '
                FROM produitsfiches AS pf LEFT JOIN stock AS s on pf.BarCode = s.BarCode
                    LEFT JOIN familles AS f ON pf.Famille = f.Famille
                WHERE pf.BarCode = '
                    . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
                GROUP BY pf.BarCode';

                $rows = $this->getFastmagSqlConnection()->get($sql);
            }
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        $this->log($this->currentJob, 'SQL parent query: ' . $sql);

        if (count($rows) > 0) {
            $row = reset($rows);

            $this->currentEntity->addData($row);
        }
    }

    /**
     * Check if products fill all the condition to be synced
     *
     * @return bool
     */
    protected function productMustBeSynced()
    {
        $result = ($this->currentEntity->getActive() && $this->currentEntity->getVisibleWeb())
            || !$this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ACTIVE_VISIBLEWEB);

        $onlyDefinedStocks = $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS);
        if ($onlyDefinedStocks && $this->currentEntity->getStockLevel() === 0) {
            $result = false;
        }

        if ($this->currentEntity->getMagentoId()) {
            $result = true;
        }

        return $result;
    }

    /**
     * Get config on scope global and store for product sync
     *
     * @return void
     */
    protected function getSyncConfig()
    {
        $defaultValues = [];
        foreach ($this->syncConfigFields as $field) {
            $defaultValues[$field] = $this->getSyncConfigValue($field);
        }

        $result['default'] = $defaultValues;

        $storesList = $this->storeRepository->getList();

        $pricePerWebsite = $this->config->getValue(Data::XML_PATH_PRICE_SCOPE);
        $this->logger->debug(
            __('[%1] Price scope: %2', $this->getCode(), ($pricePerWebsite === 0 ? 'global' : 'website'))->render()
        );

        $useMsi = $this->useMSI();
        $this->logger->debug(__('[%1] MSI: %2', $this->getCode(), ($useMsi ? 'enabled' : 'disabled'))->render());

        foreach ($storesList as $store) {
            $storeValues = [];

            foreach ($this->syncConfigFields as $field) {
                $storeValues[$field] = $this->getSyncConfigValue($field, $store->getId());

                if ($field === Config::XML_PATH_PRODUCT_PRICE_FASTMAG_RATE && $pricePerWebsite === false) {
                    $storeValues[$field] = $defaultValues[$field];
                }

                if ($useMsi === false
                    && in_array($field, [
                        Config::XML_PATH_INVENTORY_FASTMAG_STOCK_REFERENCE_STOCK,
                        Config::XML_PATH_INVENTORY_FASTMAG_STOCK_ALTERNATIVE_STOCKS
                    ], true)
                ) {
                    $storeValues[$field] = $defaultValues[$field];
                }
            }

            if ($defaultValues !== $storeValues) {
                $result['stores'][$store->getId()] = $storeValues;
            }
        }

        $this->logger->debug(
            __('[%1] Synchronization config: %2', $this->getCode(), $this->jsonSerializer->serialize($result))->render()
        );

        $this->syncConfig = $result;
    }

    /**
     * Get usefull config field value, for scope global, or for a specific store
     *
     * @param string   $field
     * @param int|null $storeId
     *
     * @return mixed
     */
    protected function getSyncConfigValue($field, $storeId = null)
    {
        $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT;

        if ($storeId !== null) {
            $scope = ScopeInterface::SCOPE_STORES;
        }

        $result = $this->config->getValue($field, $scope, $storeId);

        if ($field === Config::XML_PATH_INVENTORY_FASTMAG_STOCK_ALTERNATIVE_STOCKS) {
            if ($result === null) {
                $result = [];
            } elseif (strpos($result, '|') === false) {
                $result = [$result];
            } else {
                $result = explode('|', $result);
                sort($result);
            }
        }

        return $result;
    }

    /**
     * Check if website use MSI
     *
     * @return bool
     */
    protected function useMsi()
    {
        $searchCriteria = $this->searchCriteriaBuilder->create();

        $stocksList = $this->stockRepository->getList($searchCriteria);

        return $stocksList->getTotalCount() > 1;
    }

    /**
     * Get sync config from syncConfig array
     *
     * @param string   $field
     * @param int|null $storeId
     * @param bool     $default
     */
    protected function getSyncConfigValueFromArray($field, $storeId = null, $default = true)
    {
        $result = '';

        if ($storeId === null && $default === true) {
            $result = $this->syncConfig['default'][$field];
        }

        if ($storeId !== null && array_key_exists($storeId, $this->syncConfig['stores'])) {
            $result = $this->syncConfig['stores'][$storeId][$field];
        }

        return $result;
    }

    /**
     * Get tax rate for the product
     *
     * @param int   $storeId
     * @param float $defaultValue
     *
     * @return void
     *
     * @throws JobException
     */
    protected function getFastmagShopTaxRate($storeId, $defaultValue = 20.0)
    {
        $result = $defaultValue;

        $fastmagShop = $this->getSyncConfigValueFromArray(Config::XML_PATH_ORDER_WORKFLOW_SHOP, $storeId);

        try {
            $sql = 'SELECT TVA AS shop_vat_rate FROM mag WHERE Code = '
                . $this->getFastmagSqlConnection()->escape($fastmagShop);

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        if (count($rows) > 0) {
            $row = reset($rows);

            $result = (float)$row['shop_vat_rate'];
        }

        $currentProductI18nEntity = $this->currentEntity->getI18n($storeId);

        if ($currentProductI18nEntity !== null) {
            $currentProductI18nEntity->setShopVatRate($result);
        }
    }

    /**
     * Get static data for the product from Fastmag
     *
     * @param int|null $storeId
     *
     * @return void
     *
     * @throws JobException
     */
    protected function getChildrenData($storeId = null)
    {
        try {
            $sql = $this->getChildrenDataSql($storeId);
            $this->log($this->currentJob, 'SQL children query: ' . $sql);

            $rows = $this->getFastmagSqlConnection()->get($sql);

            if (count($rows) === 0) {
                $rows = $this->getFastmagSqlConnection()->get(str_replace('stock.AR = 1', 'stock.AR = 0', $sql));
            }
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        if (count($rows) > 0) {
            foreach ($rows as $row) {
                $this->saveChildrenData($row, $storeId);
            }
        }
    }

    /**
     * Get SQL query getting the children data
     *
     * @param int|null $storeId
     *
     * @return string
     *
     * @throws JobException
     * @throws NoConnectionException
     */
    protected function getChildrenDataSql($storeId = null)
    {
        $fastmagShop = $this->getSyncConfigValueFromArray(Config::XML_PATH_ORDER_WORKFLOW_SHOP, $storeId);
        $rateCode = $this->getSyncConfigValueFromArray(Config::XML_PATH_PRODUCT_PRICE_FASTMAG_RATE, $storeId);

        $standardPriceSubquery = $this->getStandardPriceSubquery($fastmagShop, $rateCode);
        $combinationPriceSubquery = $this->getCombinationPriceSubquery($rateCode);
        $ratePriceSubquery = $this->getRatePriceSubquery($rateCode);
        $salesPriceSubquery = $this->getSalesPriceSubquery($rateCode);
        $salesPriceChildrenSubquery = $this->getSalesPriceChildrenSubquery($fastmagShop);
        $buyingPriceSubquery = $this->getBuyingPriceSubquery();

        $stockCondition = $this->getStocksCodesCondition($storeId);

        return 'SELECT
                (
                    SELECT CONCAT(
                        \'{"\', \'fastmag_id\', \'": \', Produit,
                        \', "\', \'ean13\', \'": "\', RIGHT(GenCod, 13),
                        \'", "\', \'supplier_ref\', \'": "\', RefFournisseur,
                        \'", "\', \'weight\', \'": \', IF(PoidsTC > 0, PoidsTC, 0.1), \'}\'
                    )
                    FROM produits AS p
                    WHERE p.BarCode = stock.BarCode AND p.Taille = stock.Taille AND p.Couleur = stock.Couleur
                    LIMIT 1
                ) AS combination_data,
                (
                    SELECT ProduitActif
                    FROM produits AS p2
                    WHERE p2.BarCode = stock.BarCode AND p2.Taille = stock.Taille AND p2.Couleur = stock.Couleur
                    LIMIT 1
                ) AS active,
                (
                    SELECT 1
                    FROM stock AS s2
                    WHERE s2.BarCode = stock.BarCode AND s2.Taille = stock.Taille AND s2.Couleur = stock.Couleur
                        ' . $stockCondition . '
                    LIMIT 1
                ) AS active_stock,
                (
                    SELECT IFNULL(ValeurInteger, -1) AS no_web
                    FROM produits AS p3 LEFT JOIN complementsr AS c
                        ON p3.Produit = c.Clef AND c.Champ = \'NO_WEB\' AND c.Nature = \'PRODUITSTC\'
                    WHERE p3.BarCode = stock.BarCode AND p3.Taille = stock.Taille AND p3.Couleur = stock.Couleur
                    LIMIT 1
                ) AS no_web,
                pf.PrixVente AS indicative_price,
                (' . $standardPriceSubquery . ') AS standard_price,
                (' . $combinationPriceSubquery . ') AS combination_price,
                (' . $ratePriceSubquery . ') AS rate_price,
                (' . $salesPriceSubquery . ') AS sales_price,
                (' . $salesPriceChildrenSubquery . ') AS sales_price_children,
                (' . $buyingPriceSubquery . ') AS buying_price,
                TRIM(stock.Taille) AS size,
                TRIM(stock.Couleur) AS color,
                (
                    SELECT TRIM(Descriptif)
                    FROM produitscouleurs AS pc
                    WHERE pc.Barcode = stock.BarCode AND pc.Couleur = stock.Couleur
                ) AS color_description,
                SUM(Stock) AS stock_level,
                stock.AR AS in_stock
            FROM produitsfiches AS pf INNER JOIN stock ON pf.BarCode = stock.BarCode
            WHERE stock.AR = 1
                AND stock.BarCode =
                ' . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
                ' . $stockCondition . '
            GROUP BY stock.Barcode, stock.Taille, stock.Couleur
            HAVING active = 1 ' . $this->getOnlyDefinedStocksCondition($storeId) . '
            ORDER BY stock.Couleur, stock.Taille';
    }

    /**
     * Get the condition to use only defined stocks levels to check if a product is in stock or not
     *
     * @param int|null $storeId
     *
     * @return string
     */
    protected function getOnlyDefinedStocksCondition($storeId = null)
    {
        $result = ' AND (no_web < 1 OR no_web IS NULL)';

        if ($this->getSyncConfigValueFromArray(Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS, $storeId)) {
            $result .= ' AND active_stock = 1 ';
        }

        return $result;
    }

    /**
     * Get the condition to use stocks codes defined in config
     *
     * @param int|null $storeId
     *
     * @return string
     *
     * @throws NoConnectionException
     */
    protected function getStocksCodesCondition($storeId = null)
    {
        $result = '';

        $referenceStocks = $this->getReferenceStocks($storeId);

        if ($referenceStocks !== null && count($referenceStocks) !== 0) {
            $result = ' AND CodeMag IN (' . $this->getFastmagSqlConnection()->escape($referenceStocks) . ')';
        }

        return $result;
    }

    /**
     * Set all stock references used for synchronization
     *
     * @param int|null $storeId
     *
     * @return array
     */
    protected function getReferenceStocks($storeId = null)
    {
        $result = [];

        $result[] = $this->getSyncConfigValueFromArray(
            Config::XML_PATH_INVENTORY_FASTMAG_STOCK_REFERENCE_STOCK,
            $storeId
        );
        $alternativesStocks = $this->getSyncConfigValueFromArray(
            Config::XML_PATH_INVENTORY_FASTMAG_STOCK_ALTERNATIVE_STOCKS,
            $storeId
        );

        return array_unique(array_merge($result, $alternativesStocks));
    }

    /**
     * Save static data into specific entity
     *
     * @param array    $row
     * @param int|null $storeId
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveChildrenData($row, $storeId = null)
    {
        $row = $this->extractJsonData($row);

        $variation = $this->variationEntityFactory->create();

        if ($storeId === null) {
            $variation->addData($row);

            $this->currentEntity->addVariation($variation, $row['color'], $row['size']);
        } else {
            $currentVariation = $this->currentEntity->getVariation($row['color'], $row['size']);

            if ($currentVariation === null) {
                throw new JobException(__(
                    'No variation already existing for ref "%1", color "%2", size "%3"',
                    $this->currentEntity->getRef(),
                    $row['color'],
                    $row['size']
                ));
            }

            foreach ($row as $code => $value) {
                // Set i18n values only if it's different of the default one, except for price,
                // which has to be set for each rule to distinguish default value from no value at all
                if ($value !== ''
                    && (stripos('price', $code) !== false || $value !== $currentVariation->getData($code))
                ) {
                    $variation->setData($code, $value);
                }
            }

            if ($variation->hasData()) {
                $currentVariation->addI18n($variation, $storeId);
            }
        }
    }

    /**
     * Returns subquery for stock price (standard price)
     *
     * @param string $fastmagShop
     * @param string $priceRateCode
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getStandardPriceSubquery($fastmagShop, $priceRateCode)
    {
        try {
            if ($priceRateCode === 'prix_indicatif') {
                $result = 'pf.PrixVente';
            } else {
                $result = 'SELECT stpr.PrixVente
                    FROM stock AS stpr
                    WHERE stpr.CodeMag = ' . $this->getFastmagSqlConnection()->escape($fastmagShop) . '
                        AND stpr.AR = 1
                        AND stpr.BarCode = stock.BarCode
                        AND stpr.Couleur = stock.Couleur
                        AND stpr.Taille = stock.Taille
                        AND stpr.PrixVente > 0
                    ORDER BY stpr.Date, stpr.ID
                    LIMIT 1';
            }
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        return $result;
    }

    /**
     * Returns subquery for combination price
     *
     * @param string $priceRateCode
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getCombinationPriceSubquery($priceRateCode)
    {
        try {
            if ($priceRateCode === 'prix_indicatif') {
                $result = 'pf.PrixVente';
            } else {
                $result = 'SELECT p.PrixVente
                    FROM produits AS p
                    WHERE p.BarCode = stock.BarCode
                        AND p.Couleur = stock.Couleur
                        AND p.Taille = stock.Taille
                        AND p.PrixVente > 0';
            }
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        return $result;
    }

    /**
     * Returns subquery for rate price
     *
     * @param string $rateCode
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getRatePriceSubquery($rateCode)
    {
        try {
            if ($rateCode !== '') {
                if ($rateCode === 'prix_indicatif') {
                    $result = 'pf.PrixVente';
                } else {
                    $result = 'SELECT pt.Prix
                        FROM produitstarifs AS pt
                        WHERE pt.Tarif = ' . $this->getFastmagSqlConnection()->escape($rateCode) . '
                            AND pt.BarCode = stock.BarCode
                            AND pt.Couleur = stock.Couleur
                            AND pt.Taille = stock.Taille
                            AND pt.Prix > 0
                        ORDER BY pt.Prix DESC
                        LIMIT 1';
                }
            } else {
                $result = '0';
            }
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        return $result;
    }

    /**
     * Returns subquery for sales price
     *
     * @param string $fastmagShop
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getSalesPriceSubquery($fastmagShop)
    {
        try {
            $result = 'SELECT CONCAT(
                        \'{"\', \'sales_price\', \'": \', Prix,
                        \', "\', \'discount\', \'": \', Remise,
                        \', "\', \'begin_at\', \'": "\', DateDebut,
                        \'", "\', \'end_at\', \'": "\', DateFin,
                        \'", "\', \'reason\', \'": "\', Motif, \'"}\'
                    )
                FROM prixremise
                WHERE CodeMag = ' . $this->getFastmagSqlConnection()->escape($fastmagShop) . '
                    AND BarCode = stock.BarCode
                    AND DateDebut <= date(NOW())
                    AND DateFin >= date(NOW())
                ORDER BY DateDebut DESC, DateFin DESC, Remise, Prix
                LIMIT 1';
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        return $result;
    }

    /**
     * Returns subquery for size and color sales price
     *
     * @param string $fastmagShop
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getSalesPriceChildrenSubquery($fastmagShop)
    {
        try {
            $result = 'SELECT CONCAT(
                        \'{"\', \'sales_price\', \'": \', Prix,
                        \', "\', \'discount\', \'": \', Remise,
                        \', "\', \'begin_at\', \'": "\', DateDebut,
                        \'", "\', \'end_at\', \'": "\', DateFin,
                        \'", "\', \'reason\', \'": "\', Motif, \'"}\'
                    )
                FROM prixremisetc AS ptc
                WHERE ptc.CodeMag = ' . $this->getFastmagSqlConnection()->escape($fastmagShop) . '
                    AND ptc.BarCode = stock.BarCode
                    AND ptc.Couleur = stock.Couleur
                    AND ptc.Taille = stock.Taille
                    AND (ptc.Remise > 0 OR ptc.Prix > 0)
                    AND DateDebut <= date(NOW())
                    AND ptc.DateFin >= date(NOW())
                ORDER BY ptc.DateDebut DESC, ptc.DateFin DESC, ptc.Remise, ptc.Prix
                LIMIT 1';
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        return $result;
    }

    /**
     * Returns subquery for buying price
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getBuyingPriceSubquery()
    {
        try {
            $result = 'SELECT prod.PrixAchat
                FROM produits AS prod
                WHERE prod.BarCode = stock.BarCode
                    AND prod.Couleur = stock.Couleur
                    AND prod.Taille = stock.Taille
                LIMIT 1';
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        return $result;
    }

    /**
     * Extract JSON data and merge them to the row given
     *
     * @param array $row
     *
     * @return array
     */
    protected function extractJsonData($row)
    {
        if (array_key_exists('combination_data', $row)) {
            try {
                $combinationData = $this->jsonSerializer->unserialize($row['combination_data']);
                $row = array_merge($row, $combinationData);
                unset($row['combination_data']);
            } catch (InvalidArgumentException $exception) {
                $this->log(
                    $this->currentJob,
                    'Error when unserializing combination_data from Fastmag: ' . $row['combination_data'],
                    Monolog::WARNING
                );
            }
        }

        return $row;
    }

    /**
     * Get minimal price (indicative value) from Fastmag, if price can not be synced by other means
     *
     * @param string $fastmagRef
     *
     * @return array
     *
     * @throws JobException
     */
    protected function getMinimalPrice($fastmagRef)
    {
        try {
            $sql = 'SELECT PrixVente AS PrixVenteIndicatif
                FROM produits
                WHERE BarCode = ' . $this->getFastmagSqlConnection()->escape($fastmagRef, true) . '
                ORDER BY PrixVente
                LIMIT 1';

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        return reset($rows);
    }

    /**
     * Get all i18nized data from Fastmag
     *
     * @param int $storeId
     *
     * @return void
     *
     * @throws JobException
     */
    protected function getI18nData($storeId)
    {
        $defaultLanguage = $this->getSyncConfigValueFromArray(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_LANGUAGE);
        $language = $this->getSyncConfigValueFromArray(
            Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_LANGUAGE,
            $storeId,
            false
        );

        if ($language !== $defaultLanguage) {
            $additionalAttributes = $this->getAdditionalAttributes('pl', false);

            try {
                $sql = 'SELECT Designation AS designation,
                    Designation2 AS designation2
                    ' . $additionalAttributes . '
                FROM produitslangues AS pl
                WHERE BarCode = ' . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
                    AND TRIM(CONCAT_WS(Designation, Designation2, Modele, Matiere, Rayon, Famille, SsFamille, Saison, Descriptif, Theme)) <> \'\'
                    AND Langue = ' . $this->getFastmagSqlConnection()->escape($language, true);

                $rows = $this->getFastmagSqlConnection()->get($sql);
            } catch (Exception $exception) {
                throw new JobException(__($exception->getMessage()));
            }

            if (count($rows) > 0) {
                $row = reset($rows);

                $currentProductI18nEntity = $this->currentEntity->getI18n($storeId);
                if ($currentProductI18nEntity === null) {
                    throw new JobException(__('No data available for store #%1', $storeId));
                }

                foreach ($row as $code => $value) {
                    if (!empty($value) && $value !== $this->currentEntity->getData($code)) {
                        $currentProductI18nEntity->addData($row);
                    }
                }
            }
        }
    }

    /**
     * Returns SQL to get data for additionnal attributes, given config fields
     *
     * @param string $alias
     * @param bool   $forParent
     *
     * @return string
     */
    protected function getAdditionalAttributes($alias, $forParent = true)
    {
        $additionalAttributes = [];

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_DESCRIPTION)) {
            $additionalAttributes[] = $alias . '.Descriptif AS description';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_FAMILY)) {
            $additionalAttributes[] = 'TRIM(' . $alias . '.Famille) AS family';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SUBFAMILY)) {
            $additionalAttributes[] = 'TRIM(' . $alias . '.SsFamille) AS subfamily';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SECTION)) {
            $additionalAttributes[] = 'TRIM(' . $alias . '.Rayon) AS section';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SEASON)) {
            $additionalAttributes[] = 'TRIM(' . $alias . '.Saison) AS season';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MODEL)) {
            $additionalAttributes[] = 'TRIM(' . $alias . '.Modele) AS model';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MATERIAL)) {
            $additionalAttributes[] = 'TRIM(' . $alias . '.Matiere) AS material';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_THEME)) {
            $additionalAttributes[] = 'TRIM(' . $alias . '.Theme) AS theme';
        }

        if ($forParent) {
            $additionalAttributes = array_merge(
                $additionalAttributes,
                $this->getParentAdditionalAttributesAsArray($alias)
            );
        }

        return (count($additionalAttributes) > 0 ? ', ' . implode(', ', $additionalAttributes) : '');
    }

    /**
     * Returns as array the list of parent only additional attributes to sync, if the config allows it
     *
     * @param string $alias
     *
     * @return string[]
     */
    protected function getParentAdditionalAttributesAsArray($alias)
    {
        $result = [];

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_CREATION_DATE)) {
            $result[] = $alias . '.DateCreation AS creation_date';
        }

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_TEXTILE)) {
            $result[] = 'TRIM(' . $alias . '.Lavage) AS washing';
            $result[] = 'TRIM(' . $alias . '.Chlore) AS chlorine';
            $result[] = 'TRIM(' . $alias . '.Repassage) AS ironing';
            $result[] = 'TRIM(' . $alias . '.Pressing) AS dry_cleaning';
            $result[] = 'TRIM(' . $alias . '.Sechage) AS drying';
            $result[] = 'TRIM(' . $alias . '.AquaNettoyage) AS water';
        }

        return $result;
    }
}
