<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-25
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product;

use Exception;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product;
use Monolog\Logger as Monolog;

/**
 * Class Inventory
 *
 * Hydration class used for inserting or updating products inventories from Fastmag to Magento
 */
class Inventory extends Product
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_product_inventory';

    /** @inheritDoc */
    protected $syncConfigFields = [
        Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS,
        Config::XML_PATH_INVENTORY_FASTMAG_STOCK_REFERENCE_STOCK,
        Config::XML_PATH_INVENTORY_FASTMAG_STOCK_ALTERNATIVE_STOCKS
    ];

    /**
     * @inheritDoc
     */
    public function run()
    {
        $this->getSyncConfig();

        foreach ($this->getJobs()->getItems() as $job) {
            $this->currentJob = $job;

            $fastmagRef = $this->currentJob->getContentId();

            if ($this->getMagentoProductId($fastmagRef) !== false) {
                try {
                    $entity = $this->getDataFromFastmag();

                    $this->hydrateJob($job, $entity);
                } catch (JobException $exception) {
                    $this->invalidateJob($job, $exception);
                }

                $this->saveJob($job);
            } else {
                $this->log(
                    $job,
                    'Can not change inventory level of the product with Fastmag ref "' . $fastmagRef . '"'
                        . ', as it does not exist in Magento',
                    Monolog::NOTICE
                );
            }
        }
    }

    /**
     * @inheritDoc
     */
    protected function initiateCurrentEntity($fastmagRef)
    {
        $this->currentEntity = $this->productEntityFactory->create();
        $this->currentEntity->setRef($fastmagRef)
            ->setMagentoId($this->getMagentoProductId($fastmagRef));
    }

    /**
     * @inheritDoc
     */
    protected function getDataByStore()
    {
        foreach (array_keys($this->syncConfig['stores']) as $storeId) {
            $this->getChildrenData($storeId);
        }
    }

    /**
     * @inheritDoc
     */
    protected function getParentData()
    {
        try {
            $sql = 'SELECT pf.VisibleWeb AS visible_web,
                    pf.Actif AS active,
                    COUNT(s.ID) AS stock_level,
                    GROUP_CONCAT(DISTINCT IF(Stock >= 1, s.CodeMag, NULL)) AS stocks_list,
                    f.TauxTVA AS vat_rate
                FROM produitsfiches AS pf LEFT JOIN stock AS s on pf.BarCode = s.BarCode
                    LEFT JOIN familles AS f ON pf.Famille = f.Famille
                WHERE pf.BarCode = '
                    . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
                    AND s.AR = 1
                GROUP BY pf.BarCode';

            $rows = $this->getFastmagSqlConnection()->get($sql);

            if (count($rows) === 0) {
                $sql = 'SELECT pf.VisibleWeb AS visible_web,
                    pf.Actif AS active,
                    0 AS stock_level,
                    \'\' AS stocks_list,
                    f.TauxTVA AS vat_rate
                FROM produitsfiches AS pf LEFT JOIN stock AS s on pf.BarCode = s.BarCode
                    LEFT JOIN familles AS f ON pf.Famille = f.Famille
                WHERE pf.BarCode = '
                    . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
                GROUP BY pf.BarCode';

                $rows = $this->getFastmagSqlConnection()->get($sql);
            }
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }

        $row = reset($rows);

        $this->currentEntity->addData($row);
    }

    /**
     * @inheritDoc
     */
    protected function getChildrenDataSql($storeId = null)
    {
        $stockCondition = $this->getStocksCodesCondition($storeId);

        return 'SELECT
                (
                    SELECT CONCAT(
                        \'{"\', \'fastmag_id\', \'": \', Produit,
                        \', "\', \'ean13\', \'": "\', RIGHT(GenCod, 13),
                        \'", "\', \'supplier_ref\', \'": "\', RefFournisseur,
                        \'", "\', \'weight\', \'": \', IF(PoidsTC > 0, PoidsTC, 0.1), \'}\'
                    )
                    FROM produits AS p
                    WHERE p.BarCode = stock.BarCode AND p.Taille = stock.Taille AND p.Couleur = stock.Couleur
                    LIMIT 1
                ) AS combination_data,
                (
                    SELECT 1
                    FROM stock AS s2
                    WHERE s2.BarCode = stock.BarCode AND s2.Taille = stock.Taille AND s2.Couleur = stock.Couleur
                        ' . $stockCondition . '
                    LIMIT 1
                ) AS active_stock,
                stock.Taille AS size,
                stock.Couleur AS color,
                SUM(Stock) AS stock_level,
                stock.AR AS in_stock
            FROM produitsfiches AS pf INNER JOIN stock ON pf.BarCode = stock.BarCode
            WHERE stock.AR = 1
                AND stock.BarCode = 
                ' . $this->getFastmagSqlConnection()->escape($this->currentEntity->getRef(), true) . '
                ' . $stockCondition . '
            GROUP BY stock.Barcode, stock.Taille, stock.Couleur
            ' . $this->getOnlyDefinedStocksCondition($storeId) . '
            ORDER BY stock.Couleur, stock.Taille';
    }

    /**
     * @inheritDoc
     */
    protected function getOnlyDefinedStocksCondition($storeId = null)
    {
        $result = '';

        if ($this->getSyncConfigValueFromArray(Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS, $storeId)) {
            $result = 'HAVING active_stock = 1 ';
        }

        return $result;
    }
}
