<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-25
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Data\Rule\TaxclassfamilyInterface as TaxclassFamilyRuleModel;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\Rule\TaxclassFamilyRepositoryInterface as TaxclassFamilyRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Model\Importer;
use Fastmag\Sync\Process\Entity\ToMagento\Product as ProductEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Product\Variation as VariationEntity;
use Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;
use Magento\Catalog\Api\CategoryListInterface as CategoryList;
use Magento\Catalog\Api\CategoryRepositoryInterface as CategoryRepository;
use Magento\Catalog\Api\Data\CategoryInterface as CategoryModel;
use Magento\Catalog\Api\Data\CategoryInterfaceFactory as CategoryFactory;
use Magento\Catalog\Api\Data\ProductAttributeInterface as ProductAttribute;
use Magento\Catalog\Api\ProductAttributeRepositoryInterface as AttributeRepository;
use Magento\Catalog\Api\ProductRepositoryInterface as ProductRepository;
use Magento\Catalog\Model\Product\Type;
use Magento\Catalog\Model\ResourceModel\Category\Tree;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Eav\Api\AttributeSetRepositoryInterface as AttributeSetRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\ImportExport\Model\Import;
use Magento\ImportExport\Model\Import\ErrorProcessing\ProcessingErrorAggregatorInterface;
use Magento\Store\Api\StoreRepositoryInterface as StoreRepository;
use Magento\Store\Api\WebsiteRepositoryInterface as WebsiteRepository;
use Magento\Tax\Api\TaxClassRepositoryInterface as TaxClassRepository;
use Magento\Tax\Helper\Data as TaxHelper;
use Magento\Tax\Model\Config as TaxConfig;
use Monolog\Logger as Monolog;

/**
 * Class Save
 *
 * Integration worker for tomagento_integration_product_create and tomagento_integration_product_update jobs
 */
class Save extends Product
{
    /** @var string */
    public const PROCESS_CATEGORY_NAME = 'Produits à traiter';

    /** @inheritDoc */
    protected $code = 'tomagento_integration_product_save';

    /** @var CategoryRepository $categoryRepository */
    protected $categoryRepository;

    /** @var CategoryFactory $categoryFactory */
    protected $categoryFactory;

    /** @var CategoryList $categoryList */
    protected $categoryList;

    /** @var Importer $importer */
    protected $importer;

    /** @var AttributeSetRepository $attributeSetRepository */
    protected $attributeSetRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var Tree $tree */
    protected $tree;

    /** @var WebsiteRepository $websiteRepository */
    protected $websiteRepository;

    /** @var StoreRepository $storeRepository */
    protected $storeRepository;

    /** @var TaxClassRepository $taxClassRepository */
    protected $taxClassRepository;

    /** @var ProductRepository $productRepository */
    protected $productRepository;

    /** @var AttributeRepository $attributeRepository */
    protected $attributeRepository;

    /** @var TaxclassFamilyRepository $taxclassFamilyRuleRepository */
    protected $taxclassFamilyRuleRepository;

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /** @inheritDoc */
    protected $subordinateWorkersBefore = ['tomagento_integration_product_option'];

    /** @var array $dataToImport */
    protected $dataToImport = [];

    /** @var array $currentData */
    protected $currentData = [];

    /** @var array $currentDefaultData */
    protected $currentDefaultData = [];

    /** @var array $currentVariations */
    protected $currentVariations = [];

    /** @var Job $currentJob */
    protected $currentJob;

    /** @var ProductEntity $currentEntity */
    protected $currentEntity;

    /** @var ProductAttribute $colorAttribute */
    protected $colorAttribute;

    /** @var ProductAttribute $internalColorAttribute */
    protected $internalColorAttribute;

    /** @var ProductAttribute $sizeAttribute */
    protected $sizeAttribute;

    /** @var string $processCategoryPath */
    protected $processCategoryPath;

    /**
     * Save constructor
     *
     * @param Logger                   $logger
     * @param JobRepository            $jobRepository
     * @param Config                   $config
     * @param CategoryRepository       $categoryRepository
     * @param CategoryFactory          $categoryFactory
     * @param CategoryList             $categoryList
     * @param Importer                 $importer
     * @param AttributeSetRepository   $attributeSetRepository
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param Tree                     $tree
     * @param WebsiteRepository        $websiteRepository
     * @param StoreRepository          $storeRepository
     * @param TaxClassRepository       $taxClassRepository
     * @param ProductRepository        $productRepository
     * @param AttributeRepository      $attributeRepository
     * @param TaxclassFamilyRepository $taxclassFamilyRuleRepository
     * @param Json                     $jsonSerializer
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        CategoryRepository $categoryRepository,
        CategoryFactory $categoryFactory,
        CategoryList $categoryList,
        Importer $importer,
        AttributeSetRepository $attributeSetRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        Tree $tree,
        WebsiteRepository $websiteRepository,
        StoreRepository $storeRepository,
        TaxClassRepository $taxClassRepository,
        ProductRepository $productRepository,
        AttributeRepository $attributeRepository,
        TaxclassFamilyRepository $taxclassFamilyRuleRepository,
        Json $jsonSerializer
    ) {
        parent::__construct($logger, $jobRepository, $config);

        $this->categoryRepository = $categoryRepository;
        $this->categoryFactory = $categoryFactory;
        $this->categoryList = $categoryList;
        $this->attributeSetRepository = $attributeSetRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->tree = $tree;
        $this->websiteRepository = $websiteRepository;
        $this->storeRepository = $storeRepository;
        $this->taxClassRepository = $taxClassRepository;
        $this->productRepository = $productRepository;
        $this->attributeRepository = $attributeRepository;
        $this->taxclassFamilyRuleRepository = $taxclassFamilyRuleRepository;
        $this->jsonSerializer = $jsonSerializer;

        $this->importer = $importer;
        $this->importer->setBehavior(Import::BEHAVIOR_APPEND)
            ->setEntityCode('catalog_product')
            ->setValidationStrategy(ProcessingErrorAggregatorInterface::VALIDATION_STRATEGY_SKIP_ERRORS)
            ->setAllowedErrorCount(100)
            ->setIgnoreDuplicates(false)
            ->setCategoryPathSeparator('|');
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->error('[' . $this->getCode() . '] ' . $exception->getMessage());
            return;
        }

        try {
            $this->getColorSizeAttributesCode();
            $this->checkProcessCategory();
        } catch (ProcessException $exception) {
            foreach ($this->getJobs()->getItems() as $job) {
                $this->invalidateJob($job, $exception);
            }
        }

        foreach ($this->getJobs()->getItems() as $job) {
            $this->currentJob = $job;

            if (!$job->isInError()) {
                try {
                    $this->processJob($job);
                } catch (JobException $exception) {
                    $this->invalidateJob($job, $exception);
                }
            } else {
                $this->log($job, 'Job previously in error, skip integration.', Monolog::NOTICE);
            }
        }

        $this->runImport();

        foreach ($this->getJobs()->getItems() as $job) {
            if (!$job->isInError()) {
                $this->saveJob($job);
            }
        }
    }

    /**
     * @inheritDoc
     */
    protected function initiate()
    {
        parent::initiate();

        $defaultAttributeSetId = $this->config->getValue(Config::XML_PATH_PRODUCT_IMPORT_ATTRIBUTE_SET_ID);

        if ($defaultAttributeSetId === null) {
            throw new ProcessException(__(
                'No default attribute set defined in configuration. Please fill the config field before enabling product sync.'
            ));
        }

        try {
            $this->attributeSetRepository->get($defaultAttributeSetId);
        } catch (NoSuchEntityException $exception) {
            throw new ProcessException(__(
                'Attribute set with ID #%1 not found in Magento. Please review you config.',
                $defaultAttributeSetId
            ));
        }
    }

    /**
     * Get the color/size attributes from config and set their code for the jobs
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getColorSizeAttributesCode()
    {
        $colorAttributeId = $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_COLOR);
        $this->colorAttribute = $this->getAttribute($colorAttributeId, 'color');

        $internalColorAttributeId =
            $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_INTERNAL_COLOR);
        $this->internalColorAttribute = $this->getAttribute($internalColorAttributeId, 'internal color');

        $sizeAttributeId = $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_SIZE);
        $this->sizeAttribute = $this->getAttribute($sizeAttributeId, 'size');
    }

    /**
     * Get attribute
     *
     * @param int    $attributeId
     * @param string $label
     *
     * @return ProductAttribute
     *
     * @throws ProcessException
     */
    protected function getAttribute($attributeId, $label)
    {
        try {
            return $this->attributeRepository->get($attributeId);
        } catch (NoSuchEntityException $exception) {
            throw new ProcessException(
                __('Attribute "%1", used to save "%2" data, does not exist in Magento', $attributeId, $label)
            );
        }
    }

    /**
     * Get Fastmag process category ID, and create it if it does not exist yet
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function checkProcessCategory()
    {
        $this->searchCriteriaBuilder->addFilter(CategoryModel::KEY_NAME, self::PROCESS_CATEGORY_NAME);

        $searchCriteria = $this->searchCriteriaBuilder->create();

        $categories = $this->categoryList->getList($searchCriteria);

        if ($categories->getTotalCount() === 0) {
            $category = $this->categoryFactory->create();

            $category->setName(self::PROCESS_CATEGORY_NAME)
                ->setIncludeInMenu(false)
                ->setIsActive(false)
                ->setParentId(1);

            try {
                $category = $this->categoryRepository->save($category);
            } catch (CouldNotSaveException $exception) {
                throw new ProcessException(__($exception->getMessage()));
            }
        } else {
            $category = $categories->getItems()[0];
        }

        $this->processCategoryPath = $this->getCategoryFullPath($category);
    }

    /**
     * Get full path of a category
     *
     * @param CategoryModel $category
     *
     * @return string
     */
    protected function getCategoryFullPath($category)
    {
        $categoryTree = $this->tree->loadBreadcrumbsArray($category->getPath());

        $categoryTreePath = [];
        foreach ($categoryTree as $eachCategory) {
            $categoryTreePath[] = $eachCategory['name'];
        }

        return implode('/', $categoryTreePath);
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        $this->currentData = ['simple' => [], 'configurable' => []];
        $this->currentDefaultData = [];

        /** @var ProductEntity $entity */
        $entity = $job->getEntity();
        $this->currentEntity = $entity;

        if ($this->productMustBeSynced($entity)) {
            $this->generateDefaultData();
            $this->generateSimpleProductsData();
            $this->generateConfigurableProductData();
        } else {
            $this->log(
                $this->currentJob,
                'Product "' . $entity->getRef() . '" can\'t be synced, it must be inactive or not in stock: '
                . $this->jsonSerializer->serialize($entity->export()),
                Monolog::NOTICE
            );
        }

        $this->moveDataToImport();
    }

    /**
     * Generate default data for the entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function generateDefaultData()
    {
        $this->currentDefaultData = [
            'attribute_set_code' => $this->getDefaultAttributeSetCode(),
            'product_websites'   => $this->getWebsites(),
            'tax_class_name'     => $this->getTaxClassName()
        ];

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_CREATION_DATE)) {
            $this->currentDefaultData['created_at'] = $this->currentEntity->getCreationDate();
        }
    }

    /**
     * Returns the default attribute set code for product creation
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getDefaultAttributeSetCode()
    {
        $defaultAttributeSetId = $this->config->getValue(Config::XML_PATH_PRODUCT_IMPORT_ATTRIBUTE_SET_ID);

        try {
            $defaultAttributeSet = $this->attributeSetRepository->get($defaultAttributeSetId);

            return $defaultAttributeSet->getAttributeSetName();
        } catch (NoSuchEntityException $exception) {
            throw new JobException(__(
                'Attribute set with ID #%1 not found in Magento. Please review you config.',
                $defaultAttributeSetId
            ));
        }
    }

    /**
     * Get websites
     *
     * @return string
     */
    protected function getWebsites()
    {
        $result = '';

        $websites = $this->websiteRepository->getList();

        foreach ($websites as $website) {
            if ($website->getCode() !== 'admin') {
                $result .= $website->getCode() . ',';
            }
        }

        return trim($result, ',');
    }

    /**
     * Get default tax class name for products
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getTaxClassName()
    {
        $taxClassId = $this->config->getValue(TaxHelper::CONFIG_DEFAULT_PRODUCT_TAX_CLASS);

        if ($this->currentEntity->getVatFamily()) {
            $this->searchCriteriaBuilder->addFilter(
                TaxclassFamilyRuleModel::FAMILY,
                $this->currentEntity->getVatFamily()
            );

            $searchCriteria = $this->searchCriteriaBuilder->create();

            $taxclassFamilyRulesList = $this->taxclassFamilyRuleRepository->getList($searchCriteria);

            if ($taxclassFamilyRulesList->getTotalCount() > 0) {
                $taxclassFamilyRuleListItems = $taxclassFamilyRulesList->getItems();
                $taxclassFamilyRule = reset($taxclassFamilyRuleListItems);
                $taxClassId = $taxclassFamilyRule->getTaxClassId();
            } else {
                $this->log(
                    $this->currentJob,
                    'Product has a family with specific VAT rate in Fastmag,'
                        . ' but there\'s no equivalent tax class in Magento',
                    Monolog::WARNING
                );
            }
        }

        try {
            $taxClass = $this->taxClassRepository->get($taxClassId);

            return $taxClass->getClassName();
        } catch (NoSuchEntityException $exception) {
            throw new JobException(__('Tax class ID #%1 does not exist in Magento', $taxClassId));
        }
    }

    /**
     * Get default Fastmag attributes as an array
     *
     * @param ProductEntity $entity
     *
     * @return array
     */
    protected function getDefaultAdditionalAttributes($entity)
    {
        $result = [
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_BRAND_CODE    => $entity->getBrand(),
            Constants::ATTRIBUTE_PRODUCT_FASTMAG_SUPPLIER_CODE => $entity->getSupplier()
        ];

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_FAMILY)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_FAMILY_CODE] = $entity->getFamily();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SUBFAMILY)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_SUBFAMILY_CODE] = $entity->getSubfamily();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SECTION)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_SECTION_CODE] = $entity->getSection();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SEASON)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_SEASON_CODE] = $entity->getSeason();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MODEL)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_MODEL_CODE] = $entity->getModel();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MATERIAL)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_MATERIAL_CODE] = $entity->getMaterial();
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_THEME)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_THEME_CODE] = $entity->getTheme();
        }

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_TEXTILE)) {
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_WASHING_CODE] = $entity->getWashing();
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_CHLORINE_CODE] = $entity->getChlorine();
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_IRONING_CODE] = $entity->getIroning();
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_DRY_CLEANING_CODE] = $entity->getDryCleaning();
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_DRYING_CODE] = $entity->getDrying();
            $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_WATER_CODE] = $entity->getWater();
        }

        return $result;
    }

    /**
     * Generate data to integrate for simple products
     *
     * @return void
     *
     * @throws JobException
     */
    protected function generateSimpleProductsData()
    {
        $variations = $this->currentEntity->getVariationsFlat();

        foreach ($variations as $variation) {
            $this->generateSimpleProductData($variation);
        }
    }

    /**
     * Generate data to integrate for simple product
     *
     * @param VariationEntity $variation
     *
     * @return void
     *
     * @throws JobException
     */
    protected function generateSimpleProductData($variation)
    {
        $simpleProductData = [
            'sku'          => $this->generateSimpleProductSku($variation),
            'product_type' => Type::TYPE_SIMPLE,
            'name'         => $this->generateSimpleProductName($variation),
            'weight'       => $variation->getWeight(),
            'visibility'   => 'Not Visible Individually',
            'price'        => $this->getSimpleProductPrice($variation),
            'cost'         => $variation->getBuyingPrice(),
            'url_key'      => $this->generateUrlKey($this->generateSimpleProductName($variation)),
            'qty'          => $this->getSimpleProductStockLevel($variation),
            'additional_attributes' => $this->getSimpleProductAdditionalAttributes($variation)
        ];

        try {
            $this->productRepository->get($simpleProductData['sku']);

            if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_MANAGE_PRODUCT_STATUS)) {
                $simpleProductData['product_online'] = $this->getProductStatus($variation);
            }
        } catch (NoSuchEntityException $exception) {
            $simpleProductData['product_online'] = '0';
            $simpleProductData['categories'] = $this->processCategoryPath;
        }

        $this->currentData['simple'][$simpleProductData['sku']]['default'] = array_merge(
            $this->currentDefaultData,
            $simpleProductData,
            $this->addRequiredAttributes($variation)
        );

        $this->currentVariations[] = $this->addToCurrentVariations(
            $simpleProductData['sku'],
            $simpleProductData['additional_attributes'][$this->colorAttribute->getAttributeCode()],
            $simpleProductData['additional_attributes'][$this->sizeAttribute->getAttributeCode()]
        );

        // TODO check if i18n has actually a value changed from default values
        $i18ns = $variation->getI18ns();
        if ($i18ns !== null && count($i18ns) > 0) {
            foreach ($i18ns as $storeId => $i18n) {
                $i18nData = [
                    'sku'                => $this->generateSimpleProductSku($variation),
                    'store_view_code'    => $this->getStoreViewCode($storeId),
                    'attribute_set_code' => $this->getDefaultAttributeSetCode(),
                    'product_type'       => Type::TYPE_SIMPLE,
                    'name'               => $this->generateSimpleProductName($i18n, $storeId),
                    'price'              => $this->getSimpleProductPrice($i18n),
                    'url_key'            => $this->generateUrlKey(
                        $this->generateSimpleProductName($variation, $storeId)
                    ),
                    // Delete all default values unnecessary for store override
                    'product_websites'   => null,
                    'tax_class_name'     => null
                ];

                $this->currentData['simple'][$simpleProductData['sku']]['store'][$i18nData['store_view_code']]
                    = array_merge($this->currentDefaultData, $i18nData);
            }
        }

        $this->log(
            $this->currentJob,
            'Data to import for simple product "' . $simpleProductData['sku'] . '": '
                . $this->jsonSerializer->serialize($this->currentData['simple'][$simpleProductData['sku']])
        );
    }

    /**
     * Generate simple product SKU
     *
     * @param VariationEntity $variation
     *
     * @return string
     */
    protected function generateSimpleProductSku($variation)
    {
        $result = $this->currentEntity->getRef() . '-' . $variation->getColor() . '-' . $variation->getSize();

        if ($variation->getColor() === null) {
            $result = $this->currentEntity->getRef() . '-' . $variation->getSize();
        }

        return $result;
    }

    /**
     * Generate simple product name
     *
     * @param VariationEntity $variation
     * @param int|null        $storeId
     *
     * @return string
     */
    protected function generateSimpleProductName($variation, $storeId = null)
    {
        $result = $this->generateConfigurableProductName();

        if ($storeId !== null) {
            $parentI18n = $this->currentEntity->getI18n($storeId);

            if ($parentI18n !== null) {
                $result = $this->generateConfigurableProductName($parentI18n);
            }
        }

        if ($result !== '') {
            $result .= ' ' . $variation->getColor() . ' ' . $variation->getSize();
        }

        return trim($result);
    }

    /**
     * Generate configurable product SKU
     *
     * @return string
     */
    protected function generateConfigurableProductSku()
    {
        return $this->currentEntity->getRef();
    }

    /**
     * Generate configurable product name
     *
     * @param ProductEntity $entity
     *
     * @return string
     */
    protected function generateConfigurableProductName($entity = null)
    {
        if ($entity === null) {
            $entity = $this->currentEntity;
        }

        return trim($entity->getDesignation() . ' ' . $entity->getDesignationBis());
    }

    /**
     * Get price for a simple product
     *
     * @param VariationEntity $variation
     *
     * @return float
     */
    protected function getSimpleProductPrice($variation)
    {
        $result = $variation->getIndicativePrice();

        $standardPrice = (float)$variation->getStandardPrice();
        $combinationPrice = (float)$variation->getCombinationPrice();
        $ratePrice = (float)$variation->getRatePrice();

        if ($ratePrice > 0) {
            $result = $ratePrice;
        } elseif ($standardPrice > 0) {
            $result = $standardPrice;
        } elseif ($combinationPrice > 0) {
            $result = $combinationPrice;
        }

        return $this->formatPrice($result);
    }

    /**
     * Format price, including or excluding tax
     *
     * @param float $price
     *
     * @return float
     */
    protected function formatPrice($price)
    {
        $result = $price;

        if (!$this->config->isSetFlag(TaxConfig::CONFIG_XML_PATH_PRICE_INCLUDES_TAX)) {
            $vatRate = 20.0;

            if ($this->currentEntity->getShopVatRate()) {
                $vatRate = $this->currentEntity->getShopVatRate();
            }
            if ($this->currentEntity->getVatRate()) {
                $vatRate = $this->currentEntity->getVatRate();
            }

            $result = round($price / (1 + ($vatRate / 100)), 4);
        }

        return $result;
    }

    /**
     * Generate product URL key
     *
     * @param string $productName
     *
     * @return string
     */
    protected function generateUrlKey($productName)
    {
        return strtolower(preg_replace('#[^0-9a-z]+#i', '-', $productName));
    }

    /**
     * Get global stock level, indicated by the stock level and the token is_in_stock for each store
     *
     * @param VariationEntity $variation
     *
     * @return int
     */
    protected function getSimpleProductStockLevel($variation)
    {
        $stockLevel = ($variation->getInStock() ? $variation->getStockLevel() : 0);

        $i18ns = $variation->getI18ns();
        if ($i18ns !== null && count($i18ns) > 0) {
            foreach ($i18ns as $i18n) {
                $stockLevel += $this->getSimpleProductStockLevel($i18n);
            }
        }

        return $stockLevel;
    }

    /**
     * Get Fastmag attributes values for a simple product
     *
     * @param VariationEntity $variation
     *
     * @return array
     */
    protected function getSimpleProductAdditionalAttributes($variation)
    {
        $result = $this->getDefaultAdditionalAttributes($this->currentEntity);

        $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_PRODUCT_ID_CODE] = $variation->getFastmagId();
        $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_EAN13_CODE] = $variation->getEan13();
        $result[$this->sizeAttribute->getAttributeCode()] = $variation->getSize();

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_USE_COLOR_DESCRIPTION)) {
            $result[$this->colorAttribute->getAttributeCode()] = $variation->getColorDescription();
            $result[$this->internalColorAttribute->getAttributeCode()] = $variation->getColor();
        } else {
            $result[$this->colorAttribute->getAttributeCode()] = $variation->getColor();

            if ($variation->getColor() === ''
                && $this->config->isSetFlag(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_UNIQUE_SIZE)
            ) {
                $result[$this->colorAttribute->getAttributeCode()] =
                    $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_UNIQUE_SIZE_LABEL);
            }
        }

        return $result;
    }

    /**
     * Get product enability
     *
     * @param VariationEntity|null $variation
     *
     * @return string
     */
    protected function getProductStatus($variation = null)
    {
        $result = $this->currentEntity->getActive() && $this->currentEntity->getVisibleWeb();

        if ($result && $variation !== null) {
            $result = $variation->getActive();
        }

        return $result ? '1' : '0';
    }

    /**
     * Add to current variations list, which will be added to the configurable attribute
     *
     * @param string $sku
     * @param string $color
     * @param string $size
     *
     * @return array
     */
    protected function addToCurrentVariations($sku, $color, $size)
    {
        $result = [
            'sku' => $sku
        ];

        if (!empty($color)) {
            $result[$this->colorAttribute->getAttributeCode()] = $color;
        }
        if (!empty($size)) {
            $result[$this->sizeAttribute->getAttributeCode()] = $size;
        }

        return $result;
    }

    /**
     * Get store code by store ID
     *
     * @param int $storeId
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getStoreViewCode($storeId)
    {
        try {
            $storeView = $this->storeRepository->getById($storeId);
        } catch (NoSuchEntityException $e) {
            throw new JobException(__('Store #%1 has i18ns set in entity, but does not exist in Magento', $storeId));
        }

        return $storeView->getCode();
    }

    /**
     * Generate data to integrate for configurable product
     *
     * @return void
     *
     * @throws JobException
     */
    protected function generateConfigurableProductData()
    {
        $result = [
            'sku'          => $this->generateConfigurableProductSku(),
            'product_type' => Configurable::TYPE_CODE,
            'name'         => $this->generateConfigurableProductName(),
            'weight'       => $this->currentEntity->getWeight(),
            'visibility'   => 'Catalog, Search',
            'url_key'      => $this->generateUrlKey($this->generateConfigurableProductName()),
            'additional_attributes'   => $this->getDefaultAdditionalAttributes($this->currentEntity),
            'configurable_variations' => $this->currentVariations
        ];

        if (!$this->currentEntity->getMagentoId()) {
            $result['product_online'] = '0';
            $result['categories'] = $this->processCategoryPath;
        } elseif ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_MANAGE_PRODUCT_STATUS)) {
            $result['product_online'] = $this->getProductStatus();
        }

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_DESCRIPTION)) {
            $result['description'] = $this->currentEntity->getDescription();
            $result['short_description'] = $this->currentEntity->getDescription();
        }

        $this->currentData['configurable']['default'] = array_merge(
            $this->currentDefaultData,
            $result,
            $this->addRequiredAttributes()
        );

        // TODO check if i18n has actually a value changed from default values
        $i18ns = $this->currentEntity->getI18ns();
        if ($i18ns !== null && count($i18ns) > 0) {
            foreach ($i18ns as $storeId => $i18n) {
                $i18nData = [
                    'sku'                   => $this->generateConfigurableProductSku(),
                    'store_view_code'       => $this->getStoreViewCode($storeId),
                    'attribute_set_code'    => $this->getDefaultAttributeSetCode(),
                    'product_type'          => Configurable::TYPE_CODE,
                    'name'                  => $this->generateConfigurableProductName($i18n),
                    'url_key'               => $this->generateUrlKey($this->generateConfigurableProductName($i18n)),
                    'additional_attributes' => $this->getDefaultAdditionalAttributes($i18n),
                    // Delete all default values unnecessary for store override
                    'product_websites'      => null,
                    'tax_class_name'        => null
                ];

                if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_DESCRIPTION)) {
                    $i18nData['description'] = $i18n->getDescription();
                    $i18nData['short_description'] = $i18n->getDescription();
                }

                $i18nData = array_filter($i18nData, function ($value) {
                    if (is_array($value)) {
                        return array_filter($value);
                    } else {
                        return $value !== null && trim($value) !== '';
                    }
                });

                $this->currentData['configurable']['store'][$i18nData['store_view_code']]
                    = array_merge($this->currentDefaultData, $i18nData);
            }
        }

        $this->log(
            $this->currentJob,
            'Data to import for configurable product "' . $this->currentData['configurable']['default']['sku'] . '": '
                . $this->jsonSerializer->serialize($this->currentData['configurable'])
        );
    }

    /**
     * Add required attributes values in imported data to bypass potential exceptions during import
     *
     * @param VariationEntity $variation
     *
     * @return array
     */
    protected function addRequiredAttributes($variation = null)
    {
        return [];
    }

    /**
     * Check if there's no error on current data, then move them to the $dataToImport variable
     *
     * @return void
     */
    protected function moveDataToImport()
    {
        if (count($this->currentData['simple']) > 0) {
            if (!$this->config->isSetFlag(Config::XML_PATH_PRODUCT_CATALOG_UNIQUE_COMBINATION)
                && count($this->currentData['simple']) === 1
            ) {
                $dataMerged = $this->mergeConfigurableSimpleData();

                foreach ($dataMerged as $row) {
                    $this->dataToImport[] = $this->exportDataToImport($row);
                }
            } else {
                foreach ($this->currentData['simple'] as $row) {
                    $this->dataToImport[] = $this->exportDataToImport($row['default']);

                    if (array_key_exists('store', $row)) {
                        foreach ($row['store'] as $storeRow) {
                            $this->dataToImport[] = $this->exportDataToImport($storeRow);
                        }
                    }
                }

                $this->dataToImport[] = $this->exportDataToImport($this->currentData['configurable']['default']);

                if (array_key_exists('store', $this->currentData['configurable'])) {
                    foreach ($this->currentData['configurable']['store'] as $storeRow) {
                        $this->dataToImport[] = $this->exportDataToImport($storeRow);
                    }
                }
            }
        }
    }

    /**
     * Merge parent data in variation data
     * Only used when we don't or can't create a configurable product (config set or missing configurable attributes)
     *
     * @return array
     */
    protected function mergeConfigurableSimpleData()
    {
        $result = [];

        $childRow = reset($this->currentData['simple']);

        $result[] = $childRow['default'];

        $result[0]['visibility'] = 'Catalog, Search';

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_DESCRIPTION)) {
            $result[0]['description'] = $this->currentData['configurable']['description'];
            $result[0]['short_description'] = $this->currentData['configurable']['short_description'];
        }

        if (array_key_exists('store', $childRow)) {
            foreach ($childRow['store'] as $storeId => $storeRow) {
                $storeRow['description'] = $this->currentData['configurable']['store'][$storeId]['description'];
                $storeRow['short_description'] =
                    $this->currentData['configurable']['store'][$storeId]['short_description'];

                $result[] = $storeRow;
            }
        }

        return $result;
    }

    /**
     * Run product importer
     *
     * @throws JobException
     */
    protected function runImport()
    {
        $this->logger->debug(__(
            '[%1] Data to import: %2',
            $this->getCode(),
            $this->jsonSerializer->serialize($this->dataToImport)
        )->render());

        try {
            if (count($this->dataToImport) > 0) {
                $this->importer->processImport($this->dataToImport);
            }
        } catch (Exception $exception) {
            throw new JobException(__($exception->getMessage()));
        }
    }

    /**
     * Export data row in the format expected by the importer
     *
     * @return array
     */
    protected function exportDataToImport($row)
    {
        if (array_key_exists('additional_attributes', $row) && is_array($row['additional_attributes'])) {
            $additionnalAttributes = '';

            foreach ($row['additional_attributes'] as $code => $value) {
                $additionnalAttributes .= $code . '=' . $value . ',';
            }

            $row['additional_attributes'] = trim($additionnalAttributes, ',');
        }

        if (array_key_exists('configurable_variations', $row) && is_array($row['configurable_variations'])) {
            $configurableVariations = '';

            foreach ($row['configurable_variations'] as $variation) {
                foreach ($variation as $code => $value) {
                    $configurableVariations .= $code . '=' . $value . ',';
                }

                $configurableVariations = trim($configurableVariations, ',') . '|';
            }

            $row['configurable_variations'] = trim($configurableVariations, '|');
        }

        return $row;
    }
}
