<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-08-24
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer;

use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToFastmag\Address as AddressEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\AddressFactory as AddressEntityFactory;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Address as AbstractAddress;
use Magento\Customer\Api\AddressRepositoryInterface as AddressRepository;
use Magento\Customer\Api\CustomerRepositoryInterface as CustomerRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Sales\Api\Data\OrderInterface as OrderModel;
use Magento\Sales\Api\OrderRepositoryInterface as OrderRepository;
use Magento\Sales\Model\Order;

/**
 * Class Address
 *
 * Hydration class used for inserting or updating customers addresses from Magento to Fastmag
 */
class Address extends AbstractAddress
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_customer_address';

    /** @var OrderRepository $orderRepository */
    protected $orderRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var SortOrderBuilder $sortOrderBuilder */
    protected $sortOrderBuilder;

    /**
     * Address constructor.
     *
     * @param Logger                $logger
     * @param JobRepository         $jobRepository
     * @param Config                $config
     * @param Json                  $jsonSerializer
     * @param CustomerRepository    $customerRepository
     * @param AddressRepository     $addressRepository
     * @param AddressEntityFactory  $addressEntityFactory
     * @param OrderRepository       $orderRepository
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param SortOrderBuilder      $sortOrderBuilder
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        CustomerRepository $customerRepository,
        AddressRepository $addressRepository,
        AddressEntityFactory $addressEntityFactory,
        OrderRepository $orderRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SortOrderBuilder $sortOrderBuilder
    ) {
        parent::__construct(
            $logger,
            $jobRepository,
            $config,
            $jsonSerializer,
            $customerRepository,
            $addressRepository,
            $addressEntityFactory
        );

        $this->orderRepository = $orderRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sortOrderBuilder = $sortOrderBuilder;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        $results = $this->getDataFromMagento();

        if (is_array($results)) {
            foreach ($results as $customerId => $row) {
                $job = $this->getJob($customerId);

                if ($job !== null && !$job->isInError()) {
                    /** @var CustomerEntity $customerEntity */
                    $customerEntity = $job->getEntity();
                    $customerEntity->setBillingAddress($row['billing_address'])
                        ->setShippingAddress($row['shipping_address']);

                    $this->hydrateJob($job, $customerEntity);
                    $this->log($job, 'Customer\'s addresses hydrated');
                }
            }

            foreach ($this->getJobs()->getItems() as $job) {
                $this->saveJob($job);
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @return AddressEntity[]
     */
    protected function getDataFromMagento()
    {
        $customersIds = $this->getCustomersIds();
        $result = [];

        foreach ($customersIds as $customerId) {
            /** @var Order $lastOrder */
            $lastOrder = $this->getLastOrder($customerId);

            $billingAddress = $this->getAddressEntity($customerId, $lastOrder);
            $shippingAddress = $this->getAddressEntity($customerId, $lastOrder, 'shipping');

            $hydratedData = [
                'billing_address'  => $billingAddress,
                'shipping_address' => $shippingAddress
            ];

            $result[$customerId] = $hydratedData;
        }

        return $result;
    }

    /**
     * Returns the list of the customers IDs, if the jobs is linked to customers.
     *
     * @return int[]
     */
    protected function getCustomersIds()
    {
        return $this->getJobs()->getColumnValues(Job::CONTENT_ID);
    }

    /**
     * Get last order ID made by the customer (if the customer has made one)
     *
     * @param int $customerId
     *
     * @return OrderModel|bool
     */
    protected function getLastOrder($customerId)
    {
        $result = false;

        $sortOrder = $this->sortOrderBuilder->setField(OrderModel::CREATED_AT)
            ->setDescendingDirection()
            ->create();

        $searchCriteria = $this->searchCriteriaBuilder->addFilter(OrderModel::CUSTOMER_ID, $customerId)
            ->addSortOrder($sortOrder)
            ->setPageSize(1)
            ->create();

        $orderList = $this->orderRepository->getList($searchCriteria);

        if ($orderList->getTotalCount() === 1) {
            $result = $orderList->getItems()[0];

            $job = $this->getJob($customerId);
            $this->log($job, 'Found the last order of customer: ' . $result->getIncrementId());
        }

        return $result;
    }
}
