<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-06
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Process\Entity\ToMagento\Product as ProductEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Product\Variation as VariationEntity;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\ToMagento\Integration as IntegrationTrait;
use Magento\Catalog\Api\ProductAttributeOptionManagementInterface as AttributeOptionManager;
use Magento\Catalog\Api\ProductAttributeRepositoryInterface as AttributeRepository;
use Magento\Catalog\Api\Data\ProductAttributeInterface as Attribute;
use Magento\Eav\Api\Data\AttributeOptionInterface as AttributeOption;
use Magento\Eav\Api\Data\AttributeOptionInterfaceFactory as AttributeOptionFactory;
use Magento\Eav\Api\Data\AttributeOptionLabelInterfaceFactory as AttributeOptionLabelFactory;
use Magento\Framework\Exception\NoSuchEntityException;

/**
 * Class Option
 *
 * Integration worker setting attributes option before saving products
 */
class Option extends Worker
{
    use IntegrationTrait;

    /** @inheritDoc */
    protected $code = 'tomagento_integration_product_option';

    /** @var AttributeRepository $attributeRepository */
    protected $attributeRepository;

    /** @var AttributeOptionManager $attributeOptionManager */
    protected $attributeOptionManager;

    /** @var AttributeOptionFactory $attributeOptionFactory */
    protected $attributeOptionFactory;

    /** @var AttributeOptionLabelFactory $attributeOptionLabelFactory */
    protected $attributeOptionLabelFactory;

    /** @var array $attributesToSync */
    protected $attributesToSync;

    /** @var bool $useColorDescrption */
    protected $useColorDescrption;

    /**
     * Customer constructor.
     *
     * @param Logger                      $logger
     * @param JobRepository               $jobRepository
     * @param Config                      $config
     * @param AttributeRepository         $attributeRepository
     * @param AttributeOptionManager      $attributeOptionManager
     * @param AttributeOptionFactory      $attributeOptionFactory
     * @param AttributeOptionLabelFactory $attributeOptionLabelFactory
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        AttributeRepository $attributeRepository,
        AttributeOptionManager $attributeOptionManager,
        AttributeOptionFactory $attributeOptionFactory,
        AttributeOptionLabelFactory $attributeOptionLabelFactory
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->config = $config;
        $this->attributeRepository = $attributeRepository;
        $this->attributeOptionManager = $attributeOptionManager;
        $this->attributeOptionFactory = $attributeOptionFactory;
        $this->attributeOptionLabelFactory = $attributeOptionLabelFactory;

        $this->useColorDescrption =
            $this->config->isSetFlag(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_USE_COLOR_DESCRIPTION);
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->notice($exception->getMessage());

            return;
        }

        $this->setAttributesToSync();

        foreach ($this->getJobs()->getItems() as $job) {
            if (!$job->isInError()) {
                try {
                    $this->processJob($job);
                } catch (JobException $exception) {
                    $this->invalidateJob($job, $exception);
                }

                $this->saveJob($job);
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ENABLE);
    }

    /**
     * Set the attributes list to sync
     *
     * @return void
     */
    protected function setAttributesToSync()
    {
        $result = [
            'brand' => Constants::ATTRIBUTE_PRODUCT_FASTMAG_BRAND_CODE,
            'supplier' => Constants::ATTRIBUTE_PRODUCT_FASTMAG_SUPPLIER_CODE
        ];

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_FAMILY)) {
            $result['family'] = Constants::ATTRIBUTE_PRODUCT_FASTMAG_FAMILY_CODE;
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SUBFAMILY)) {
            $result['subfamily'] = Constants::ATTRIBUTE_PRODUCT_FASTMAG_SUBFAMILY_CODE;
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SECTION)) {
            $result['section'] = Constants::ATTRIBUTE_PRODUCT_FASTMAG_SECTION_CODE;
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SEASON)) {
            $result['season'] = Constants::ATTRIBUTE_PRODUCT_FASTMAG_SEASON_CODE;
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MODEL)) {
            $result['model'] = Constants::ATTRIBUTE_PRODUCT_FASTMAG_MODEL_CODE;
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MATERIAL)) {
            $result['material'] = Constants::ATTRIBUTE_PRODUCT_FASTMAG_MATERIAL_CODE;
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_THEME)) {
            $result['theme'] = Constants::ATTRIBUTE_PRODUCT_FASTMAG_THEME_CODE;
        }

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_TEXTILE)) {
            $result['washing'] = $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_WASHING_CODE];
            $result['chlorine'] = $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_CHLORINE_CODE];
            $result['ironing'] = $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_IRONING_CODE];
            $result['dry_cleaning'] = $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_DRY_CLEANING_CODE];
            $result['drying'] = $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_DRYING_CODE];
            $result['water'] = $result[Constants::ATTRIBUTE_PRODUCT_FASTMAG_WATER_CODE];
        }

        $this->attributesToSync = $result;
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        /** @var ProductEntity $entity */
        $entity = $job->getEntity();

        if ($this->productMustBeSynced($entity)) {
            foreach ($this->attributesToSync as $fieldCode => $attributeCode) {
                $defaultValue = $entity->getData($fieldCode);

                $this->saveOption($attributeCode, $defaultValue);
            }

            foreach ($entity->getVariationsFlat() as $variation) {
                $this->saveColorSizeOptions($variation);
            }
        }
    }

    /**
     * Check if products fill all the condition to be synced
     *
     * @param ProductEntity $entity
     *
     * @return bool
     */
    protected function productMustBeSynced($entity)
    {
        $result = ($entity->getActive() && $entity->getVisibleWeb())
            || !$this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ACTIVE_VISIBLEWEB);

        $onlyDefinedStocks = $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS);
        if ($onlyDefinedStocks && $entity->getStockLevel() === 0) {
            $result = false;
        }

        if ($entity->getMagentoId()) {
            $result = true;
        }

        return $result;
    }

    /**
     * Create or update option value with all the i18nized labels
     *
     * @todo Get options labels to check if there are differencies with the labels we want to save
     *
     * @param int|string $attributeCode
     * @param string     $defaultValue
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveOption($attributeCode, $defaultValue)
    {
        try {
            $attribute = $this->attributeRepository->get($attributeCode);
        } catch (NoSuchEntityException $exception) {
            throw new JobException(__('Attribute "%1" does not exist in Magento', $attributeCode));
        }

        if (!empty($defaultValue)) {
            $option = $this->getOption($attribute, $defaultValue);

            if (!$option) {
                $option = $this->attributeOptionFactory->create();
                $option->setLabel($defaultValue);

                try {
                    $this->attributeOptionManager->add($attributeCode, $option);
                } catch (Exception $exception) {
                    throw new JobException(
                        __('Exception when adding option "%1" of the attribute "%2"', $defaultValue, $attributeCode)
                    );
                }
            }
        }
    }

    /**
     * Check if option already exists for attribute
     *
     * @param Attribute $attribute
     * @param string    $defaultValue
     *
     * @return AttributeOption|false
     */
    protected function getOption($attribute, $defaultValue)
    {
        $result = false;

        if ($attribute->usesSource()) {
            $options = $attribute->getOptions();

            foreach ($options as $option) {
                if ($option->getLabel() === $defaultValue) {
                    $result = $option;
                    break;
                }
            }
        }

        return $result;
    }

    /**
     * Save color and size options given the config of a product's variation
     *
     * @param VariationEntity $entity
     *
     * @throws JobException
     */
    protected function saveColorSizeOptions($entity)
    {
        $colorAttributeId = $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_COLOR);

        if ($this->useColorDescrption) {
            $this->saveOption($colorAttributeId, $entity->getData('color_description'));

            $this->saveOption(
                $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_INTERNAL_COLOR),
                $entity->getData('color')
            );
        } else {
            $this->saveOption($colorAttributeId, $entity->getData('color'));
        }

        $this->saveOption(
            $this->config->getValue(Config::XML_PATH_PRODUCT_CONFIGURABLE_ATTRIBUTES_SIZE),
            $entity->getData('size')
        );
    }
}
