<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-18
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag;

use DateInterval;
use DateTime;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Jobqueue as AbstractJob;
use Fastmag\Sync\Model\Jobqueue\ToFastmagRepository as JobRepository;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\Clean as CleanTrait;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Clean
 *
 * Worker class used for cleaning to Fastmag jobs finished since more than 45 minutes
 */
class Clean extends Worker
{
    use CleanTrait;

    /** @inheritDoc */
    protected $code = 'tofastmag_clean';

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /**
     * Clean constructor.
     *
     * @param Logger                $logger
     * @param ResourceConnection    $resourceConnection
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param JobRepository         $jobRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        JobRepository $jobRepository
    ) {
        parent::__construct($logger);

        $this->resourceConnection = $resourceConnection;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     *
     * @throws CouldNotSaveException
     * @throws CouldNotDeleteException
     */
    public function run()
    {
        $this->resetErrorJobs();
        $this->cleanDoneJobs();
    }

    /**
     * Update jobs data to allow rerun
     *
     * @throws CouldNotSaveException
     */
    protected function resetErrorJobs()
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(Job::RETRY_COUNT, 4, 'lt')
            ->addFilter(Job::STATUS, AbstractJob::STATUS_ERROR)
            ->create();

        $jobsToReset = $this->jobRepository->getList($searchCriteria);

        foreach ($jobsToReset->getItems() as $job) {
            $this->jobRepository->reset($job);
            $this->jobRepository->save($job);
        }
    }

    /**
     * Clean done jobs from queue
     *
     * @throws CouldNotDeleteException
     */
    protected function cleanDoneJobs()
    {
        $dateTimeLimit = new DateTime();
        $dateTimeLimit->sub(new DateInterval('PT45M'));

        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(Job::STATUS, AbstractJob::STATUS_OK)
            ->addFilter(Job::PROCESSED_AT, $dateTimeLimit->format('Y-m-d H:i:s'), 'lt')
            ->create();

        $jobsToDelete = $this->jobRepository->getList($searchCriteria);

        foreach ($jobsToDelete->getItems() as $job) {
            $this->jobRepository->delete($job);
        }

        $this->setLastExecutionDate();
    }
}
