<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-20
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration;

use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Data\OrderInterface as SyncedOrder;
use Fastmag\Sync\Api\Data\OrderInterfaceFactory as SyncedOrderFactory;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Data\Rule\PaymentcodeInterface as PaymentcodeRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\OrderRepositoryInterface as SyncedOrderRepository;
use Fastmag\Sync\Api\Rule\OrdertransactionRepositoryInterface as OrdertransactionRuleRepository;
use Fastmag\Sync\Api\Rule\PaymentcodeRepositoryInterface as PaymentcodeRuleRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\OrderFactory as OrderEntityFactory;
use Fastmag\Sync\Process\Worker\FastmagSql as FastmagSqlTrait;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration as HydrationTrait;
use Magento\Catalog\Model\Product\Type;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\Area;
use Magento\Framework\App\State;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Sales\Api\Data\OrderInterface as OrderModel;
use Magento\Sales\Api\OrderRepositoryInterface as OrderRepository;
use Magento\Store\Model\ScopeInterface;

/**
 * Class Order
 *
 * Hydration class used for inserting or updating orders from Magento to Fastmag
 */
class Order extends Worker
{
    use HydrationTrait;
    use FastmagSqlTrait;

    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_order';

    /** @var OrderRepository $orderRepository */
    protected $orderRepository;

    /** @var SyncedOrderRepository $syncedOrderRepository */
    protected $syncedOrderRepository;

    /** @var SyncedOrderFactory $syncedOrderFactory */
    protected $syncedOrderFactory;

    /** @var OrdertransactionRuleRepository $ordertransactionRuleRepository */
    protected $ordertransactionRuleRepository;

    /** @var PaymentcodeRuleRepository $paymentcodeRuleRepository */
    protected $paymentcodeRuleRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var OrderEntityFactory $orderEntityFactory */
    protected $orderEntityFactory;

    /** @var State $state */
    protected $state;

    /** @var Job $currentJob */
    protected $currentJob;

    /** @var OrderEntity $currentEntity */
    protected $currentEntity;

    /** @var OrdertransactionRule[] $ordertransactionRules */
    protected $ordertransactionRules;

    /** @var PaymentcodeRule[] $paymentcodeRules */
    protected $paymentcodeRules;

    /** @var string[] $subordinateWorkersAfter */
    protected $subordinateWorkersAfter = ['tofastmag_hydration_order_customer', 'tofastmag_hydration_order_address'];

    /**
     * Order constructor.
     *
     * @param Logger                         $logger
     * @param JobRepository                  $jobRepository
     * @param Config                         $config
     * @param Json                           $jsonSerializer
     * @param FastmagSql                     $fastmagSql
     * @param OrderRepository                $orderRepository
     * @param SyncedOrderRepository          $syncedOrderRepository
     * @param SyncedOrderFactory             $syncedOrderFactory
     * @param OrdertransactionRuleRepository $ordertransactionRuleRepository
     * @param PaymentcodeRuleRepository      $paymentcodeRuleRepository
     * @param SearchCriteriaBuilder          $searchCriteriaBuilder
     * @param OrderEntityFactory             $orderEntityFactory
     * @param State                          $state
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        FastmagSql $fastmagSql,
        OrderRepository $orderRepository,
        SyncedOrderRepository $syncedOrderRepository,
        SyncedOrderFactory $syncedOrderFactory,
        OrdertransactionRuleRepository $ordertransactionRuleRepository,
        PaymentcodeRuleRepository $paymentcodeRuleRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        OrderEntityFactory $orderEntityFactory,
        State $state
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->config = $config;
        $this->jsonSerializer = $jsonSerializer;
        $this->fastmagSql = $fastmagSql;
        $this->orderRepository = $orderRepository;
        $this->syncedOrderRepository = $syncedOrderRepository;
        $this->syncedOrderFactory = $syncedOrderFactory;
        $this->ordertransactionRuleRepository = $ordertransactionRuleRepository;
        $this->paymentcodeRuleRepository = $paymentcodeRuleRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->orderEntityFactory = $orderEntityFactory;
        $this->state = $state;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        $this->state->setAreaCode(Area::AREA_ADMINHTML);

        $this->getRules();

        foreach ($this->getJobs()->getItems() as $job) {
            $this->currentJob = $job;
            $this->currentEntity = $this->orderEntityFactory->create();

            try {
                $this->getDataFromMagento();

                $this->hydrateJob($job, $this->currentEntity);
            } catch (JobException $exception) {
                $this->invalidateJob($job, $exception);
            }

            $this->saveJob($job);
        }
    }

    /**
     * Get rules
     *
     * @return void
     */
    protected function getRules()
    {
        $criteria = $this->searchCriteriaBuilder->create();

        $this->ordertransactionRules = $this->ordertransactionRuleRepository->getList($criteria)->getItems();
        $this->paymentcodeRules = $this->paymentcodeRuleRepository->getList($criteria)->getItems();
    }

    /**
     * @inheritDoc
     *
     * @return void
     *
     * @throws JobException
     */
    protected function getDataFromMagento()
    {
        [$orderId, $status] = $this->getOrderDataFromJobContentId($this->currentJob->getContentId());

        $this->currentEntity->setMagentoId($orderId)
            ->setStatus($status);

        $order = $this->orderRepository->get($orderId);
        $paymentMethod = $this->getOrderPaymentMethod($order);
        $lastTransaction = $this->getLastTransaction($orderId);
        $transactionType = $this->getTransactionType($paymentMethod, $status, $lastTransaction);

        if ($transactionType === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_NONE) {
            throw new JobException(__(
                'No transaction type found for order #%1. Order status: %2 - Payment method: %3',
                $order->getIncrementId(),
                $status,
                $paymentMethod
            ));
        }

        $this->setConfigData($order->getStoreId());

        $this->currentEntity->setIncrementId($order->getIncrementId())
            ->setCustomerId($order->getCustomerId())
            ->setGrandTotal($order->getGrandTotal())
            ->setPaymentCode($this->getFastmagPaymentCode($paymentMethod))
            ->setLastTransaction($lastTransaction)
            ->setTransactionType($transactionType)
            ->setIsExcludedTax($this->isOrderExcludingTax($order, $transactionType))
            ->setTotalQty($this->getTotalQuantity($order, $transactionType));

        $items = $this->getOrderItems($order, $transactionType);

        foreach ($items as $item) {
            $this->currentEntity->addItem($item);
        }

        $this->addFastmagDataToCurrentEntity();

        $shippingData = $this->getShippingData($order, $transactionType);

        if (count($shippingData) > 0) {
            $this->currentEntity->setShippingRate($shippingData['shipping_rate'])
                ->setShippingDiscount($shippingData['shipping_discount'])
                ->setShippingDescription($shippingData['shipping_description'])
                ->setPaymentId($this->getPaymentId($order, $transactionType));
        }
    }

    /**
     * Retrieve order payment method
     *
     * @param OrderModel $order
     *
     * @return string
     */
    protected function getOrderPaymentMethod($order)
    {
        $result = '*';

        if ($order->getPayment()) {
            $result = $order->getPayment()->getMethod();
        }

        return $result;
    }

    /**
     * Get last transaction synced with Fastmag
     *
     * @param int $orderId
     *
     * @return SyncedOrder|null
     */
    protected function getLastTransaction($orderId)
    {
        try {
            $result = $this->syncedOrderRepository->getLastSyncByOrderId($orderId);
        } catch (LocalizedException $exception) {
            $result = null;
        }

        return $result;
    }

    /**
     * Get most specific order/transaction rule given the payment method and the status of the order.
     *
     * @param string           $paymentMethod
     * @param string           $orderStatus
     * @param SyncedOrder|null $lastTransaction
     *
     * @return string
     */
    protected function getTransactionType($paymentMethod, $orderStatus, $lastTransaction)
    {
        $transactionType = $this->getTransactionTypeByRule($paymentMethod, $orderStatus);

        if ($transactionType === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE
            && $lastTransaction !== null
            && $lastTransaction->getTransactionType() === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION
        ) {
            $transactionType = OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE;
        } elseif ($transactionType === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_CANCELLATION
            && $lastTransaction !== null
        ) {
            if ($lastTransaction->getTransactionType() === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION) {
                $transactionType = OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION_CANCELLATION;
            } elseif ($lastTransaction->getTransactionType() === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE) {
                $transactionType = OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE_CANCELLATION;
            }
        }

        if ($lastTransaction === null
            && in_array($transactionType, [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE_CANCELLATION,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION_CANCELLATION
            ], true)
        ) {
            $transactionType = OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_NONE;
        }

        return $transactionType;
    }

    /**
     * Get most specific order/transaction rule given the payment method and the status of the order.
     *
     * @param string $paymentMethod
     * @param string $orderStatus
     *
     * @return string
     */
    protected function getTransactionTypeByRule($paymentMethod, $orderStatus)
    {
        $result = OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_NONE;
        $bestRule = null;

        foreach ($this->ordertransactionRules as $rule) {
            $rulePaymentMethod = $rule->getPaymentMethod();
            $ruleOrderStatus = $rule->getOrderStatus();

            if ($bestRule === null && $rulePaymentMethod === '*' && $ruleOrderStatus === '*') {
                $bestRule = $rule;
            }
            if (($bestRule === null || $bestRule->getOrderStatus() === '*')
                && $rulePaymentMethod === '*'
                && $ruleOrderStatus === $orderStatus
            ) {
                $bestRule = $rule;
            }
            if ($rulePaymentMethod === $paymentMethod && $ruleOrderStatus === $orderStatus) {
                $bestRule = $rule;
            }
        }

        if ($bestRule && $bestRule->getTransactionType() !== null) {
            $result = $bestRule->getTransactionType();
        }

        if ($bestRule !== null) {
            $this->log(
                $this->currentJob,
                'Best order/transaction rule found. Payment method: "' . $paymentMethod . '".'
                . ' Rule #' . $bestRule->getId() . ', transaction type: ' . $result
            );
        } else {
            $this->log(
                $this->currentJob,
                'No order/transaction rule found. Payment method: "' . $paymentMethod . '"'
            );
        }

        return $result;
    }

    /**
     * Set Fastmag chain, shop, seller and stock according to config
     *
     * @param int $storeId
     *
     * @return void
     *
     * @throws JobException
     */
    protected function setConfigData($storeId)
    {
        $this->currentEntity->setFastmagChain(
            $this->config->getValue(Config::XML_PATH_ORDER_WORKFLOW_CHAIN, ScopeInterface::SCOPE_STORES, $storeId)
        );
        $this->currentEntity->setFastmagShop(
            $this->config->getValue(Config::XML_PATH_ORDER_WORKFLOW_SHOP, ScopeInterface::SCOPE_STORES, $storeId)
        );
        $this->currentEntity->setFastmagSeller(
            $this->config->getValue(Config::XML_PATH_ORDER_WORKFLOW_SELLER, ScopeInterface::SCOPE_STORES, $storeId)
        );

        $this->currentEntity->setFastmagReferenceStock(
            $this->config->getValue(
                Config::XML_PATH_INVENTORY_FASTMAG_STOCK_REFERENCE_STOCK,
                ScopeInterface::SCOPE_STORES,
                $storeId
            )
        );

        if (!$this->currentEntity->getFastmagChain()
            || !$this->currentEntity->getFastmagShop()
            || !$this->currentEntity->getFastmagSeller()
            || !$this->currentEntity->getFastmagReferenceStock()
        ) {
            throw new JobException(__('Missing chain/shop/seller/stock configuration, unable to sync order.'));
        }

        $this->currentEntity->setFastmagAlternativeStocks(
            $this->config->getValue(
                Config::XML_PATH_INVENTORY_FASTMAG_STOCK_ALTERNATIVE_STOCKS,
                ScopeInterface::SCOPE_STORES,
                $storeId
            )
        );

        $this->currentEntity->setSendStatus(
            $this->config->getValue(
                Config::XML_PATH_ORDER_WORKFLOW_FASTMAG_STATUS,
                ScopeInterface::SCOPE_STORES,
                $storeId
            )
        );
    }

    /**
     * Check if the order with $orderId has not been already sent to Fastmag.
     * Workaround to avoid duplication in case of Fastmag EDI crash during order sync.
     *
     * @param int    $orderId
     * @param string $transactionType
     *
     * @return bool
     *
     * @throws JobException
     */
    protected function checkTransactionDuplicate($orderId, $transactionType)
    {
        $result = false;

        if (in_array(
            $transactionType,
            [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION
            ],
            true
        )) {
            $isOrderSyncedInMagento = (bool)$this->syncedOrderRepository->getListByOrderId($orderId)->getTotalCount();

            try {
                $sql = 'SELECT v.Vente AS transaction_id, v.Date AS date, v.Heure AS hour, v.CodeMag AS target_shop
                    FROM ventes
                    WHERE v.Nature = ' . $this->getFastmagSqlConnection()->escape($transactionType) . '
                        AND v.InfosComp LIKE ' . $this->getFastmagSqlConnection()->escape($orderId) . '
                    ORDER BY v.Vente DESC
                    LIMIT 1';

                $row = $this->getFastmagSqlConnection()->get($sql);
            } catch (NoConnectionException $exception) {
                throw new JobException(__(
                    'Error when trying to check order ID in Fastmag for duplicates. Message: %1. Order ID: %2 - Transaction type: %3',
                    $exception->getMessage(),
                    $orderId,
                    $transactionType
                ));
            }

            if (!$isOrderSyncedInMagento && count($row) > 0) {
                $result = true;

                $fastmagTransactionData = $row[0];
                $syncedOrder = $this->syncedOrderFactory->create();

                $syncedOrder->setOrderId($orderId)
                    ->setTransactionId($fastmagTransactionData['transaction_id'])
                    ->setTransactionType($transactionType)
                    ->setState('OK')
                    ->setTargetShop($fastmagTransactionData['target_shop'])
                    ->setRequestAt($fastmagTransactionData['date'] . ' ' . $fastmagTransactionData['hour'])
                    ->setResultAt($fastmagTransactionData['date'] . ' ' . $fastmagTransactionData['hour']);

                try {
                    $this->syncedOrderRepository->save($syncedOrder);
                } catch (CouldNotSaveException $exception) {
                    throw new JobException(__(
                        'Error when trying to save synced order during first check. Message: %1. Order ID: %2 - Transaction type: %3',
                        $exception->getMessage(),
                        $orderId,
                        $transactionType
                    ));
                }
            }
        }

        return $result;
    }

    /**
     * Check if order is including or excluding tax
     *
     * @param OrderModel $order
     * @param string     $transactionType
     *
     * @return bool
     *
     * @todo: find a better way to find out if an order is taxed or not
     */
    protected function isOrderExcludingTax($order, $transactionType)
    {
        $result = false;

        if (in_array(
            $transactionType,
            [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_ORDER
            ],
            true
        )) {
            $result = $order->getTaxAmount() <= 0;
        }

        return $result;
    }

    /**
     * Get order data (ID, state and status) from job content ID
     *
     * @param int $contentId
     *
     * @return array
     */
    protected function getOrderDataFromJobContentId($contentId)
    {
        return explode('_', $contentId);
    }

    /**
     * Get order ID from job content ID
     *
     * @param int $contentId
     *
     * @return int
     */
    protected function getOrderIdFromJobContentId($contentId)
    {
        $orderData = $this->getOrderDataFromJobContentId($contentId);

        return $orderData[0];
    }

    /**
     * Retrieve items data
     *
     * @param OrderModel $order
     * @param string     $transactionType
     *
     * @return array
     */
    protected function getOrderItems($order, $transactionType)
    {
        $result = [];

        if (!in_array(
            $transactionType,
            [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_NONE
            ],
            true
        )) {
            foreach ($order->getItems() as $item) {
                if ($item->getProductType() === Type::TYPE_SIMPLE) {
                    $result[] = $item;
                }
            }
        }

        return $result;
    }

    /**
     * Retrieve total quantity of the order (ordered or cancelled)
     *
     * @param OrderModel $order
     * @param string     $transactionType
     *
     * @return int
     */
    protected function getTotalQuantity($order, $transactionType)
    {
        $result = 0;

        if ($transactionType === OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_CANCELLATION) {
            foreach ($order->getItems() as $item) {
                $result += $item->getQtyCanceled();
            }
        } else {
            $result = $order->getTotalQtyOrdered();
        }

        return $result;
    }

    /**
     * Get basic data from Fastmag
     *
     * @return void
     *
     * @throws JobException
     */
    protected function addFastmagDataToCurrentEntity()
    {
        $itemsFastmagIds = [];

        $items = $this->currentEntity->getItems();

        foreach ($items as $itemId => $item) {
            $itemsFastmagIds[$item->getFastmagId()] = $itemId;
        }

        try {
            $sql = 'SELECT Produit AS product_id, BarCode AS barcode, Taille AS size, Couleur AS color
                FROM produits
                WHERE Produit IN (' . $this->getFastmagSqlConnection()->escape(array_keys($itemsFastmagIds)) . ')';

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (NoConnectionException $exception) {
            throw new JobException(__(
                'Error when trying to check order\'s items product ids in Fastmag. Message: %1. Product IDs: %3',
                $exception->getMessage(),
                array_keys($itemsFastmagIds)
            ));
        }

        if (count($rows) < count($items)) {
            throw new JobException(__(
                'Unable to find corresponding products in Fastmag for some items of the order #%1',
                $this->currentEntity->getMagentoId()
            ));
        }

        foreach ($rows as $row) {
            $itemEntity = $this->currentEntity->getItemByFastmagId($row['product_id']);

            if ($itemEntity !== null) {
                $itemEntity->setFastmagBarcode($row['barcode'])
                    ->setFastmagSize($row['size'])
                    ->setFastmagColor($row['color']);
            }
        }
    }

    /**
     * Get shipping rate data
     *
     * @param OrderModel $order
     * @param string     $transactionType
     *
     * @return array
     */
    protected function getShippingData($order, $transactionType)
    {
        $result = [];

        if (!in_array(
            $transactionType,
            [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_CANCELLATION,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_NONE
            ],
            true
        )) {
            $result = [
                'shipping_rate'        => $order->getBaseShippingInclTax(),
                'shipping_discount'    => $order->getBaseShippingDiscountAmount(),
                'shipping_description' => ($order->getShippingDescription() ?? __('SHIPPING'))
            ];
        }
        
        return $result;
    }

    /**
     * Get payment data
     *
     * @param OrderModel $order
     * @param string     $transactionType
     *
     * @return string
     */
    protected function getPaymentId($order, $transactionType)
    {
        $result = '';

        if (in_array(
            $transactionType,
            [
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE,
                OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_ORDER
            ],
            true
        )) {
            $payment = $order->getPayment();

            if ($payment !== null) {
                $result = $payment->getEntityId();

                if ($payment->getCcTransId()) {
                    $result = $payment->getCcTransId();
                } elseif ($payment->getLastTransId()) {
                    $result = $payment->getLastTransId();
                }
            }
        }

        return $result;
    }

    /**
     * Get Fastmag payment code
     *
     * @param string $paymentMethod
     *
     * @return string
     */
    protected function getFastmagPaymentCode($paymentMethod)
    {
        $result = '';
        $bestRule = null;

        foreach ($this->paymentcodeRules as $rule) {
            if ($rule->getPaymentMethod() === $paymentMethod) {
                $bestRule = $rule;
            }
            if ($bestRule === null && $rule->getPaymentMethod() === '*') {
                $bestRule = $rule;
            }
        }

        if ($bestRule && $bestRule->getFastmagCode() !== null) {
            $result = $bestRule->getFastmagCode();
        }

        if ($bestRule !== null) {
            $this->log(
                $this->currentJob,
                'Best payment/code rule found. Payment method: "' . $paymentMethod . '".'
                . ' Rule #' . $bestRule->getId() . ', Fastmag code: ' . $result
            );
        } else {
            $this->log(
                $this->currentJob,
                'No payment/code rule found. Payment method: "' . $paymentMethod . '"'
            );
        }

        return $result;
    }
}
