<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-18
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration\Order;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Data\OrderInterfaceFactory as SyncOrderFactory;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\OrderRepositoryInterface as SyncOrderRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Helper\Data as Helper;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Api;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\Order\Item as OrderItemEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration\Transfer;
use Fastmag\Sync\Process\Worker\FastmagSql as FastmagSqlTrait;
use Magento\Framework\Serialize\Serializer\Json;
use stdClass;

/**
 * Class OrderTransfer
 *
 * Integration worker used for sending inventory transfer in Fastmag
 */
class OrderTransfer extends Transfer
{
    use FastmagSqlTrait;

    /** @inheritDoc */
    protected $code = 'tofastmag_integration_order_transfer';

    /** @var SyncOrderFactory $syncOrderFactory */
    protected $syncOrderFactory;

    /** @var SyncOrderRepository $syncOrderRepository */
    protected $syncOrderRepository;

    /** @var OrderEntity $currentEntity */
    protected $currentEntity;

    /**
     * OrderTransfer constructor
     *
     * @param Logger              $logger
     * @param JobRepository       $jobRepository
     * @param Api                 $api
     * @param Config              $config
     * @param Json                $jsonSerializer
     * @param FastmagSql          $fastmagSql
     * @param SyncOrderFactory    $syncOrderFactory
     * @param SyncOrderRepository $syncOrderRepository
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Api $api,
        Config $config,
        Json $jsonSerializer,
        FastmagSql $fastmagSql,
        SyncOrderFactory $syncOrderFactory,
        SyncOrderRepository $syncOrderRepository
    ) {
        parent::__construct($logger, $jobRepository, $api, $config, $jsonSerializer);

        $this->fastmagSql = $fastmagSql;
        $this->syncOrderFactory = $syncOrderFactory;
        $this->syncOrderRepository = $syncOrderRepository;
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        /** @var OrderEntity $orderEntity */
        $this->currentEntity = $job->getEntity();

        $transactionType = $this->currentEntity->getTransactionType();

        if ($this->transactionTypeNeedsTransfers($transactionType)) {
            $itemsStockLevels = $this->checkForTransfers();

            if (count($itemsStockLevels) > 0) {
                foreach ($itemsStockLevels as $itemId => $itemStockLevels) {
                    $this->sendTransfers($itemId, $itemStockLevels);
                }
            }
        }
    }

    /**
     * Tells if a transaction type needs inventory transfers before
     *
     * @param string $transactionType
     *
     * @return bool
     */
    protected function transactionTypeNeedsTransfers($transactionType)
    {
        return in_array($transactionType, [
            OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE,
            OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION
        ], true);
    }

    /**
     * Check on Fastmag if the quantity ordered require stock transfers on Fastmag DB
     *
     * @return array
     *
     * @throws JobException
     */
    protected function checkForTransfers()
    {
        $result = [];

        $stockCodes = $this->currentEntity->getAllFastmagStocks();

        $items = $this->currentEntity->getItems();

        if (count($items) > 0) {
            foreach ($items as $itemId => $item) {
                $qtyOrdered = $item->getQtyOrdered();
                $barcode = $item->getFastmagBarcode();
                $size = $item->getFastmagSize();
                $color = $item->getFastmagColor();

                $inventoryLevels = $this->getFastmagInventoryLevels($barcode, $size, $color, $stockCodes);

                if ($inventoryLevels[$stockCodes[0]] < $qtyOrdered) {
                    $result[$itemId] = $inventoryLevels;
                }
            }
        }

        return $result;
    }

    /**
     * Get Fastmag inventory levels for the item
     *
     * @param string   $barcode
     * @param string   $size
     * @param string   $color
     * @param string[] $stockCodes
     *
     * @return array
     *
     * @throws JobException
     */
    protected function getFastmagInventoryLevels($barcode, $size, $color, $stockCodes)
    {
        $result = [];
        foreach ($stockCodes as $stockCode) {
            $result[$stockCode] = 0;
        }

        try {
            $conditionSizeColor = $this->getConditionSizeColor($size, $color);

            $sql = 'SELECT CodeMag AS stock_code, SUM(Stock) AS total_inventory
                FROM stock
                WHERE CodeMag IN (' . $this->getFastmagSqlConnection()->escape($stockCodes) . ')
                    AND AR = 1
                    AND BarCode = ' . $this->getFastmagSqlConnection()->escape($barcode) . '
                    ' . $conditionSizeColor . '
                GROUP BY CodeMag
                HAVING SUM(Stock) > 0
                LIMIT 1';

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new JobException(__(
                'Error when getting inventory level of order\'s items. Message: %2. Item barcode: %2. Item size: %3. Item color: %4. Stock code: %5',
                $exception->getMessage(),
                $barcode,
                $size,
                $color,
                implode(',', $stockCodes)
            ));
        }

        if (count($rows) > 0) {
            foreach ($rows as $row) {
                $result[$row['stock_code']] = (int)$row['total_inventory'];
            }
        }

        return $result;
    }

    /**
     * Get SQL condition of size and color when getting inventory levels on Fastmag
     *
     * @param string $size
     * @param string $color
     *
     * @return string
     *
     * @throws NoConnectionException
     */
    protected function getConditionSizeColor($size, $color)
    {
        $result = '';

        if ($size !== null) {
            $result .= ' AND Taille = ' . $this->getFastmagSqlConnection()->escape($size);
        }
        if ($color !== null) {
            $result .= ' AND Couleur = ' . $this->getFastmagSqlConnection()->escape($color);
        }

        return $result;
    }

    /**
     * Send transfers to Fastmag
     *
     * @param int   $itemId
     * @param array $itemStockLevels
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendTransfers($itemId, $itemStockLevels)
    {
        $itemEntity = $this->currentEntity->getItemByMagentoId($itemId);

        if ($itemEntity === null || count($this->currentEntity->getItems()) === 0) {
            throw new JobException(
                __('Item #%1 can not be transferred as it does not exist in the order in the first place', $itemId)
            );
        }

        $qtyToTransfer = $itemEntity->getQtyOrdered();
        $qtyTransferred = 0;
        $shop = $this->currentEntity->getFastmagShop();

        foreach ($itemStockLevels as $stockCode => $inventoryLevel) {
            if ($stockCode !== $shop && $inventoryLevel >= $qtyToTransfer) {
                $transferData = [
                    'qty_to_transfer' => $qtyToTransfer,
                    'origin_shop'     => $stockCode
                ];

                $this->sendTransfer($itemEntity, $transferData);

                $qtyToTransfer = 0;
                break;
            }
        }

        if ($qtyToTransfer > 0) {
            foreach ($itemStockLevels as $stockCode => $inventoryLevel) {
                if ($stockCode !== $shop && $inventoryLevel > 0) {
                    $qtyTransferred = min($qtyToTransfer, $inventoryLevel);
                    $transferData = [
                        'qty_to_transfer' => $qtyTransferred,
                        'origin_shop'     => $stockCode
                    ];

                    $this->sendTransfer($itemEntity, $transferData);

                    $qtyToTransfer -= $qtyTransferred;
                }

                if ($qtyToTransfer === 0) {
                    break;
                }
            }
        }
    }

    /**
     * Generate API request for transfer
     *
     * @param OrderItemEntity $itemEntity
     * @param array           $transferData
     *
     * @return stdClass
     */
    protected function generateTransferDataRequest($itemEntity, $transferData)
    {
        $result = [
            'MagasinOrigine'     => $transferData['origin_shop'],
            'MagasinDestination' => $this->currentEntity->getFastmagShop(),
            'TotalQuantite'      => (int)$transferData['qty_to_transfer'],
            'Vendeur'            => $this->currentEntity->getFastmagSeller(),
            'MAJStock' => $this->config->isSetFlag(Config::XML_PATH_ORDER_WORKFLOW_INVENTORY_TRANSFER)? 'OUI' : 'NON',
            'Produits' => [
                (object)[
                    'Reference'           => $itemEntity->getFastmagBarcode(),
                    'Couleur'             => $itemEntity->getFastmagSize(),
                    'Taille'              => $itemEntity->getFastmagColor(),
                    'Designation'         => Helper::removeAccents($itemEntity->getName()),
                    'PrixUnitaire'        => Helper::formatPrice($itemEntity->getPriceInclTax()),
                    'Quantite'            => (int)$transferData['qty_to_transfer'],
                    'TotalLigneTransfert' => Helper::formatPrice(
                        $itemEntity->getPriceInclTax() * $transferData['qty_to_transfer']
                    ),
                ]
            ]
        ];

        return (object)['Transferts' => [(object)$result]];
    }
}
