<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-18
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration;

use Exception;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Api;
use Fastmag\Sync\Process\Entity\ToFastmag\Order\Item as OrderItemEntity;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration as IntegrationTrait;
use Magento\Framework\Serialize\Serializer\Json;

abstract class Transfer extends Worker
{
    use IntegrationTrait;

    /** @var Config $config */
    protected $config;

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /**
     * Transfer constructor
     *
     * @param Logger        $logger
     * @param JobRepository $jobRepository
     * @param Api           $api
     * @param Config        $config
     * @param Json          $jsonSerializer
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Api $api,
        Config $config,
        Json $jsonSerializer
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->api = $api;
        $this->config = $config;
        $this->jsonSerializer = $jsonSerializer;
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_ORDER_SYNC_ENABLE);
    }

    /**
     * Send transfer to Fastmag
     *
     * @param OrderItemEntity $itemEntity
     * @param array           $transferData
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendTransfer($itemEntity, $transferData)
    {
        $endpoint = '/boa/transfert/create/index.ips';
        $request = $this->generateTransferDataRequest($itemEntity, $transferData);

        try {
            $requestAt = date('Y-m-d H:i:s');

            $response = $this->api->post($endpoint, $request);
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to send inventory transfer for order #%1 on Fastmag through API. Code: %2. Message: %3. Data sent: %4',
                $this->currentEntity->getMagentoId(),
                $exception->getCode(),
                $exception->getMessage(),
                $exception->getRequest()
            ));
        }

        if (array_key_exists('data', $response)) {
            $response = $response['data'][0];
        }

        if ($response['status'] === 'OK') {
            if ($this->config->isSetFlag(Config::XML_PATH_ORDER_ALERT_ENABLE_FOR_TRANSFER)) {
                $this->sendAlert(
                    $transferData['origin_shop'],
                    [$itemEntity->getMagentoId() => $itemEntity],
                    [$itemEntity->getMagentoId() => $transferData['qty_to_transfer']]
                );
            }

            $syncedOrder = $this->syncOrderFactory->create();

            $syncedOrder->setOrderId($this->currentEntity->getMagentoId())
                ->setTransactionId($response['Id'])
                ->setTransactionType(OrdertransactionInterface::FASTMAG_TRANSACTION_TYPE_TRANSFER)
                ->setRequest($this->jsonSerializer->serialize($request))
                ->setEndpoint($endpoint)
                ->setStatus($response['status'])
                ->setRequestAt($requestAt)
                ->setResultAt(date('Y-m-d H:i:s'))
                ->setOriginShop($transferData['origin_shop'])
                ->setTargetShop($this->currentEntity->getFastmagShop());

            try {
                $this->syncOrderRepository->save($syncedOrder);
            } catch (Exception $exception) {
                throw new JobException(
                    __('Unable to save order #%1 synchronization on Magento DB', $this->currentEntity->getMagentoId())
                );
            }
        }
    }

    /**
     * Send alert to shop
     *
     * @param string            $originShop
     * @param OrderItemEntity[] $items
     * @param array             $qtyTransferred
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendAlert($originShop, $items, $qtyTransferred = [])
    {
        $message = [];

        foreach ($items as $itemId => $itemEntity) {
            $message[] = 'Ref : ' . $itemEntity->getFastmagBarcode()
                . ' - Taille : ' . $itemEntity->getFastmagSize()
                . ' - Couleur : ' . $itemEntity->getFastmagColor()
                . ' - Quantité : ';

            if (array_key_exists($itemId, $qtyTransferred)) {
                $message[] .= $qtyTransferred[$itemId];
            } else {
                $message[] .= $itemEntity->getQtyOrdered();
            }
        }

        $request = (object)[
            'Consignes' => [(object)[
                'Destinataire' => $this->currentEntity->getFastmagShop(),
                'Sujet'        => __('Web order #%1', $this->currentEntity->getMagentoId()),
                'Message'      => __('To prepare: %1', implode(' *** ', $message)),
                'Expediteur'   => $originShop,
                'Slide'        => '1'
            ]]
        ];

        try {
            $response = $this->api->post('/boa/consigne/create/index.ips', $request);

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                throw new ApiException(__($message), $this->api->getLastRequest(), null, $response['errorCode']);
            }
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to send alert to shop %1. Code: %2. Message: %3. Alert sent: %4',
                $this->currentEntity->getFastmagShop(),
                $exception->getCode(),
                $exception->getMessage(),
                implode(' *** ', $message)
            ));
        }
    }
}
