<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-24
 ******************************************************************************/

namespace Fastmag\Sync\Model;

use Exception;
use Fastmag\Sync\Api\Data\OrderInterface;
use Fastmag\Sync\Api\Data\OrderInterfaceFactory;
use Fastmag\Sync\Api\Data\OrderSearchResultsInterfaceFactory;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as Ordertransaction;
use Fastmag\Sync\Api\OrderRepositoryInterface;
use Fastmag\Sync\Model\ResourceModel\Order as ResourceModel;
use Fastmag\Sync\Model\ResourceModel\Order\CollectionFactory;
use Magento\Framework\Api\DataObjectHelper;
use Magento\Framework\Api\SearchCriteria\CollectionProcessor\FilterProcessor;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Reflection\DataObjectProcessor;

/**
 * Class OrderRepository
 *
 * Repository class for synchronized orders model
 */
class OrderRepository implements OrderRepositoryInterface
{
    /** @var ResourceModel $resource */
    protected $resource;

    /** @var OrderFactory $orderFactory */
    protected $orderFactory;

    /** @var CollectionFactory $collectionFactory */
    protected $collectionFactory;

    /** @var OrderSearchResultsInterfaceFactory $searchResultsFactory */
    protected $searchResultsFactory;

    /** @var DataObjectHelper $dataObjectHelper */
    protected $dataObjectHelper;

    /** @var DataObjectProcessor $dataObjectProcessor */
    protected $dataObjectProcessor;

    /** @var OrderInterfaceFactory $orderInterfaceFactory */
    protected $orderInterfaceFactory;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var SortOrderBuilder $sortOrderBuilder */
    protected $sortOrderBuilder;

    /** @var CollectionProcessorInterface $collectionProcessor */
    private $collectionProcessor;

    /**
     * OrderRepository constructor
     *
     * @param ResourceModel                      $resource
     * @param OrderFactory                       $orderFactory
     * @param OrderInterfaceFactory              $orderInterfaceFactory
     * @param CollectionFactory                  $collectionFactory
     * @param OrderSearchResultsInterfaceFactory $searchResultsFactory
     * @param DataObjectHelper                   $dataObjectHelper
     * @param DataObjectProcessor                $dataObjectProcessor
     * @param SearchCriteriaBuilder              $searchCriteriaBuilder
     * @param SortOrderBuilder                   $sortOrderBuilder
     * @param CollectionProcessorInterface|null  $collectionProcessor
     */
    public function __construct(
        ResourceModel $resource,
        OrderFactory $orderFactory,
        OrderInterfaceFactory $orderInterfaceFactory,
        CollectionFactory $collectionFactory,
        OrderSearchResultsInterfaceFactory $searchResultsFactory,
        DataObjectHelper $dataObjectHelper,
        DataObjectProcessor $dataObjectProcessor,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SortOrderBuilder $sortOrderBuilder,
        CollectionProcessorInterface $collectionProcessor = null
    ) {
        $this->resource = $resource;
        $this->orderFactory = $orderFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->dataObjectHelper = $dataObjectHelper;
        $this->orderInterfaceFactory = $orderInterfaceFactory;
        $this->dataObjectProcessor = $dataObjectProcessor;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sortOrderBuilder = $sortOrderBuilder;
        $this->collectionProcessor = $collectionProcessor ?: $this->getCollectionProcessor();
    }

    /**
     * @inheritDoc
     */
    public function save(OrderInterface $order)
    {
        try {
            $this->resource->save($order);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $order;
    }

    /**
     * @inheritDoc
     */
    public function getById($orderId)
    {
        $order = $this->orderFactory->create();
        $this->resource->load($order, $orderId);
        if (!$order->getId()) {
            throw new NoSuchEntityException(__('The order with the "%1" ID doesn\'t exist.', $orderId));
        }

        return $order;
    }

    /**
     * @inheritDoc
     */
    public function getLastSyncByOrderId(
        $orderId,
        $excludeTypes = [Ordertransaction::FASTMAG_TRANSACTION_TYPE_TRANSFER]
    ) {
        $sortOrderResultAt = $this->sortOrderBuilder->setField(OrderInterface::RESULT_AT)
            ->setDescendingDirection()
            ->create();

        $sortOrderEntityId = $this->sortOrderBuilder->setField(OrderInterface::SYNC_ID)
            ->setDescendingDirection()
            ->create();

        $this->searchCriteriaBuilder->addFilter(OrderInterface::ORDER_ID, $orderId)
            ->addSortOrder($sortOrderResultAt)
            ->addSortOrder($sortOrderEntityId);

        if (count($excludeTypes) > 0) {
            $this->searchCriteriaBuilder->addFilter(OrderInterface::TRANSACTION_TYPE, $excludeTypes, 'nin');
        }

        $criteria = $this->searchCriteriaBuilder->create();

        $syncList = $this->getList($criteria);

        if ($syncList->getTotalCount() === 0) {
            throw new LocalizedException(__('There\'s no line on synced orders for the order #%1', $orderId));
        }

        $items = $syncList->getItems();

        return array_shift($items);
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $criteria)
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($criteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function getListByOrderId($orderId)
    {
        $criteria = $this->searchCriteriaBuilder->addFilter(OrderInterface::ORDER_ID, $orderId)
            ->create();

        return $this->getList($criteria);
    }

    /**
     * @inheritDoc
     */
    public function delete(OrderInterface $order)
    {
        try {
            $this->resource->delete($order);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($entityId)
    {
        return $this->delete($this->getById($entityId));
    }

    /**
     * Retrieve collection processor
     *
     * @deprecated
     *
     * @return CollectionProcessorInterface
     */
    private function getCollectionProcessor()
    {
        if (!$this->collectionProcessor) {
            $this->collectionProcessor = ObjectManager::getInstance()->get(FilterProcessor::class);
        }

        return $this->collectionProcessor;
    }
}
