<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-27
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration\Order;

use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Data\OrderInterfaceFactory as SyncOrderFactory;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\OrderRepositoryInterface as SyncOrderRepository;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Helper\Data as Helper;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Api;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\Order\Item as OrderItemEntity;
use Fastmag\Sync\Process\Worker\FastmagSql as FastmagSqlTrait;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration\Order;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Serialize\Serializer\Json;
use stdClass;

/**
 * Class Data
 *
 * Integration class used for inserting or updating orders from Magento to Fastmag
 */
class Data extends Order
{
    use FastmagSqlTrait;

    /** @inheritDoc */
    protected $code = 'tofastmag_integration_order';

    /** @var string $hydrationWorker */
    protected $hydrationWorker = 'tofastmag_hydration_order';

    /** @var string[] $subordinateWorkersBefore */
    protected $subordinateWorkersBefore = [
        'tofastmag_integration_order_customer',
        'tofastmag_integration_order_address',
        'tofastmag_integration_order_transfer'
    ];

    /** @var string[] $subordinateWorkersAfter */
    protected $subordinateWorkersAfter = [
        'tofastmag_integration_order_cancellation_transfer'
    ];

    /**
     * Integration constructor.
     *
     * @param Logger                $logger
     * @param JobRepository         $jobRepository
     * @param Api                   $api
     * @param Config                $config
     * @param SyncOrderFactory      $syncedOrderFactory
     * @param SyncOrderRepository   $syncedOrderRepository
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param Json                  $jsonSerializer
     * @param FastmagSql            $fastmagSql
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Api $api,
        Config $config,
        SyncOrderFactory $syncedOrderFactory,
        SyncOrderRepository $syncedOrderRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        Json $jsonSerializer,
        FastmagSql $fastmagSql
    ) {
        parent::__construct(
            $logger,
            $jobRepository,
            $api,
            $config,
            $syncedOrderFactory,
            $syncedOrderRepository,
            $searchCriteriaBuilder,
            $jsonSerializer
        );

        $this->fastmagSql = $fastmagSql;
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        /** @var OrderEntity $orderEntity */
        $this->currentEntity = $job->getEntity();
        $this->currentJobApiCalls = [];

        $transactionType = $this->currentEntity->getTransactionType();

        $requestAt = date('Y-m-d H:i:s');

        switch ($transactionType) {
            case OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE:
                $this->sendSale();
                break;
            case OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION:
                $this->sendReservation();
                break;
            case OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_ORDER:
                $this->sendOrder();
                break;
            case OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE:
                $this->sendReservationToSale();
                break;
            case OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE_CANCELLATION:
            case OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION_CANCELLATION:
                $this->sendCancellation();
                break;
            default:
                throw new JobException(__('No transaction type for the current order'));
        }

        $this->saveSyncOrder($requestAt);
    }

    /**
     * Send ReservationToSale transaction to Fastmag
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendReservationToSale()
    {
        $endpoint = '/boa/resavente/create/index.ips';
        $request = $this->generateReservationToSaleDataRequest();

        try {
            $this->sendRequest(OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATIONTOSALE, $endpoint, $request);
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to send reservation to sale for order #%1 on Fastmag through API. Code: %2. Message: %3. Data sent: %4',
                $this->currentEntity->getMagentoId(),
                $exception->getCode(),
                $exception->getMessage(),
                $exception->getRequest()
            ));
        }

        if ($this->currentEntity->getSendStatus()) {
            $request =
                [(object)['Transactions' => [(object)[
                    'Numero'  => $this->currentEntity->getLastTransaction()->getTransactionId(),
                    'Magasin' => $this->currentEntity->getFastmagShop(),
                    'Statut'  => '100-Commande reçue'
                ]]]];

            try {
                $response = $this->api->post('/boa/sale/vshopstatus/index.ips', $request);

                if ($response['status'] === 'KO') {
                    $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                    throw new ApiException(__($message), $this->api->getLastRequest(), null, $response['errorCode']);
                }
            } catch (ApiException $exception) {
                throw new JobException(__(
                    'Unable to send reservation to sale for order #%1 on Fastmag through API. Code: %2. Message: %3. Data sent: %4',
                    $this->currentEntity->getMagentoId(),
                    $exception->getCode(),
                    $exception->getMessage(),
                    $exception->getRequest()
                ));
            }
        }
    }

    /**
     * Send sale to Fastmag
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendSale()
    {
        $endpoint = '/boa/sale/create/index.ips';
        $request = $this->generateSaleDataRequest();

        try {
            $this->sendRequest(OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE, $endpoint, $request);
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to send sale #%1 on Fastmag through API. Code: %2. Message: %3. Data sent: %4',
                $this->currentEntity->getMagentoId(),
                $exception->getCode(),
                $exception->getMessage(),
                $exception->getRequest()
            ));
        }

        if ($this->config->isSetFlag(Config::XML_PATH_ORDER_ALERT_ENABLE_FOR_SALE)) {
            $this->sendAlert($this->currentEntity->getItems());
        }
    }

    /**
     * Send reservation to Fastmag
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendReservation()
    {
        $endpoint = '/boa/Reservation/create/index.ips';
        $request = $this->generateReservationDataRequest();

        try {
            $this->sendRequest(OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION, $endpoint, $request);
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to send reservation #%1 on Fastmag through API. Code: %2. Message: %3. Data sent: %4',
                $this->currentEntity->getMagentoId(),
                $exception->getCode(),
                $exception->getMessage(),
                $exception->getRequest()
            ));
        }
    }

    /**
     * Send order to Fastmag
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendOrder()
    {
        $endpoint = '/boa/order/create/index.ips';
        $request = $this->generateOrderDataRequest();

        try {
            $this->sendRequest(OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_ORDER, $endpoint, $request);
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to send order #%1 on Fastmag through API. Code: %2. Message: %3. Data sent: %4',
                $this->currentEntity->getMagentoId(),
                $exception->getCode(),
                $exception->getMessage(),
                $exception->getRequest()
            ));
        }
    }

    /**
     * Generate API request for reservation to sale
     *
     * @return stdClass
     *
     * @throws JobException
     */
    protected function generateReservationToSaleDataRequest()
    {
        $message = 'Unable to transform reservation to sale for order #'
            . $this->currentEntity->getMagentoId() . ' to Fastmag: ';

        if ($this->currentEntity->getCustomer() === null) {
            throw new JobException(__($message . 'no customer found.'));
        }

        if ($this->currentEntity->getShippingAddress() === null) {
            throw new JobException(__($message . 'no shipping address found.'));
        }

        if ($this->currentEntity->getLastTransaction() === null) {
            throw new JobException(__($message . 'no previous transaction found.'));
        }

        $totalQty = (int)$this->currentEntity->getTotalQty();
        if ($this->currentEntity->getShippingRate() > 0) {
            ++$totalQty;
        }

        $result = [
            'TotalResa'        => $this->currentEntity->getGrandTotal(),
            'TotalQuantite'    => $totalQty,
            'Vendeur'          => $this->currentEntity->getFastmagSeller(),
            'Magasin'          => $this->currentEntity->getFastmagShop(),
            'Email'            => $this->currentEntity->getCustomer()->getFastmagId(),
            'AdresseLivraison' => $this->currentEntity->getShippingAddress()->getAlias(),
            'NumResa'          => $this->currentEntity->getLastTransaction()->getTransactionId(),
            'Reglements'       => [
                (object)[
                    'CodeReglement' => $this->currentEntity->getPaymentCode(),
                    'Montant'       => $this->currentEntity->getGrandTotal()
                ]
            ]
        ];

        return (object)['Reservations' => [(object)$result]];
    }

    /**
     * Generate API request for sale
     *
     * @return stdClass
     *
     * @throws JobException
     */
    protected function generateSaleDataRequest()
    {
        $result = $this->generateGenericTransactionData();

        $result['Produits'] = $this->generateSaleItemsData();

        $result['reglements'] = [(object)[
            'CodeReglement'     => $this->currentEntity->getPaymentCode(),
            'Montant'           => Helper::formatPrice($this->currentEntity->getGrandTotal()),
            'ReferencePaiement' => $this->currentEntity->getPaymentId()
        ]];

        return (object)['Ventes' => [(object)$result]];
    }

    /**
     * Generate API request for reservation
     *
     * @return stdClass
     *
     * @throws JobException
     */
    protected function generateReservationDataRequest()
    {
        $result = $this->generateGenericTransactionData();

        $result['Produits'] = $this->generateGenericTransactionItemsData();

        return (object)['Reservations' => [(object)$result]];
    }

    /**
     * Generate API request for order
     *
     * @return stdClass
     *
     * @throws JobException
     */
    protected function generateOrderDataRequest()
    {
        $result = $this->generateGenericTransactionData();

        $result['Produits'] = $this->generateGenericTransactionItemsData();

        return (object)['Commandes' => [(object)$result]];
    }

    /**
     * Generate API request data for generic transaction
     *
     * @return array
     *
     * @throws JobException
     */
    protected function generateGenericTransactionData()
    {
        if ($this->currentEntity->getCustomer() === null) {
            throw new JobException(__(
                'Unable to send transaction for order #%1 to Fastmag through EDI: no customer found.',
                $this->currentEntity->getMagentoId()
            ));
        }
        if ($this->currentEntity->getShippingAddress() === null) {
            throw new JobException(__(
                'Unable to send transaction for order #%1 to Fastmag through EDI: no shipping address found.',
                $this->currentEntity->getMagentoId()
            ));
        }

        $totalQty = (int)$this->currentEntity->getTotalQty();
        if ($this->currentEntity->getShippingRate() > 0) {
            ++$totalQty;
        }

        $result = [
            'TotalVenteTTC'    => Helper::formatPrice($this->currentEntity->getGrandTotal()),
            'TotalQuantite'    => $totalQty,
            'Vendeur'          => $this->currentEntity->getFastmagSeller(),
            'Magasin'          => $this->currentEntity->getFastmagShop(),
            'Email'            => $this->currentEntity->getCustomer()->getFastmagId(),
            'AdresseLivraison' => $this->currentEntity->getShippingAddress()->getAlias(),
            'InfosComp'        => 'Magento ID: #' . $this->currentEntity->getIncrementId(),
            'Provenance'       => 'VSHOP'
        ];

        if ($this->currentEntity->getSendStatus()) {
            $result['EtatPiece'] = 'ESHOPSTATUS';
        }

        return $result;
    }

    /**
     * Generate API request product data for generic transaction
     *
     * @return stdClass[]
     */
    protected function generateGenericTransactionItemsData()
    {
        $result = [];

        foreach ($this->currentEntity->getItems() as $itemEntity) {
            $itemData = [
                'Reference'     => '#' . $itemEntity->getFastmagId(),
                'Designation'   => Helper::removeAccents($itemEntity->getName()),
                'PrixUnitaire'  => Helper::formatPrice($itemEntity->getPriceInclTax()),
                'Remise'        => Helper::formatPrice($itemEntity->getDiscountAmount()),
                'Quantite'      => (int)$itemEntity->getQtyOrdered(),
                'TotalLigneTTC' => Helper::formatPrice($itemEntity->getRowTotalInclTax())
            ];

            $result[] = (object)$itemData;
        }

        if ($this->currentEntity->getShippingRate() > 0) {
            $result[] = (object)[
                'Reference'     => 'FP',
                'Description'   => $this->currentEntity->getShippingDescription(),
                'PrixUnitaire'  => $this->currentEntity->getShippingRate(),
                'Remise'        => $this->currentEntity->getShippingDiscount(),
                'Quantite'      => 1,
                'TotalLigneTTC' => $this->currentEntity->getShippingRate()
            ];
        }

        return $result;
    }

    /**
     * Generate API request product data for sale
     *
     * @return array
     */
    protected function generateSaleItemsData()
    {
        $result = $this->generateGenericTransactionItemsData();

        foreach ($result as $itemData) {
            $total = $itemData->TotalLigneTTC;
            $itemData->TotalLigneVente = $total;
            unset($itemData->TotalLigneTTC);
        }

        return $result;
    }

    /**
     * Send alert to shop
     *
     * @param OrderItemEntity[] $items
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendAlert($items)
    {
        $message = [];

        foreach ($items as $itemEntity) {
            $message[] = 'Ref : ' . $itemEntity->getFastmagBarcode()
                . ' - Taille : ' . $itemEntity->getFastmagSize()
                . ' - Couleur : ' . $itemEntity->getFastmagColor()
                . ' - Nom : ' . $itemEntity->getName()
                . ' - Quantité : ' . $itemEntity->getQtyOrdered();
        }

        $request = (object)[
            'Consignes' => [(object)[
                'Destinataire' => $this->currentEntity->getFastmagShop(),
                'Sujet'        => __('Web order #%1', $this->currentEntity->getMagentoId()),
                'Message'      => __('To prepare: %1', implode(' *** ', $message)),
                'Expediteur'   => $this->currentEntity->getFastmagShop(),
                'Slide'        => '1'
            ]]
        ];

        try {
            $response = $this->api->post('/boa/consigne/create/index.ips', $request);

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                throw new ApiException(__($message), $this->api->getLastRequest(), null, $response['errorCode']);
            }
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to send alert to shop %1. Code: %2. Message: %3. Alert sent: %4',
                $this->currentEntity->getFastmagShop(),
                $exception->getCode(),
                $exception->getMessage(),
                implode(' *** ', $message)
            ));
        }
    }
}
