<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-28
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Fastmag\Sync\Process\Entity\ToMagento\OrderFactory as OrderEntityFactory;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * Class Order
 *
 * Hydration class used for inserting and updating orders from Fastmag to Magento
 */
class Order extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_order';

    /** @var OrderEntityFactory $orderEntityFactory */
    protected $orderEntityFactory;

    /**
     * Integration constructor.
     *
     * @param Logger             $logger
     * @param JobRepository      $jobRepository
     * @param Config             $config
     * @param Json               $jsonSerializer
     * @param OrderEntityFactory $orderEntityFactory
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        Proxy $fastmagSql,
        OrderEntityFactory $orderEntityFactory
    ) {
        parent::__construct($logger, $jobRepository, $config, $jsonSerializer, $fastmagSql);

        $this->orderEntityFactory = $orderEntityFactory;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $results = $this->getDataFromFastmag();

            foreach ($this->getJobs()->getItems() as $job) {
                if (array_key_exists($job->getContentId(), $results)) {
                    $this->hydrateJob($job, $results[$job->getContentId()]);
                } else {
                    $exception = new JobException(__('No syncable status update found'));
                    $this->skipJob($job, $exception);
                }

                $this->saveJob($job);
            }
        } catch (ProcessException $exception) {
            foreach ($this->getJobs()->getItems() as $job) {
                $this->invalidateJob($job, $exception);
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    protected function getDataFromFastmag()
    {
        $ordersIds = $this->getJobs()->getColumnValues(Job::CONTENT_ID);

        try {
            $sql = 'SELECT Vente AS order_id, VShopStatus AS status, ListeColis AS tracking_number
                FROM ventes AS v
                WHERE v.Vente IN (' . $this->getFastmagSqlConnection()->escape($ordersIds) . ')
                    AND TRIM(ListeColis) <> \'\'';

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new ProcessException(__(
                'Error when hydrating orders. Message: %1. Orders IDs: %2',
                $exception->getMessage(),
                implode(', ', $ordersIds)
            ));
        }

        $result = [];
        foreach ($rows as $row) {
            $orderEntity = $this->orderEntityFactory->create();
            $orderEntity->setFastmagId($row['order_id'])
                ->setStatus($row['status'])
                ->setTrackingNumber($row['tracking_number']);

            $result[$row['order_id']] = $orderEntity;
        }

        return $result;
    }
}
