<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-28
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration\Order;

use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\OrderRepositoryInterface as SyncedOrderRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToMagento\Order as OrderEntity;
use Fastmag\Sync\Process\Worker\ToMagento\Integration\Order;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Sales\Api\OrderRepositoryInterface as OrderRepository;
use Magento\Sales\Api\ShipmentRepositoryInterface as ShipmentRepository;
use Magento\Sales\Exception\CouldNotShipException;
use Magento\Sales\Model\Convert\Order as OrderConverter;
use Magento\Sales\Model\Order as OrderModel;
use Magento\Sales\Model\Order\Shipment as ShipmentModel;
use Magento\Sales\Model\Order\Shipment\TrackFactory;
use Magento\Shipping\Model\ShipmentNotifier;

/**
 * Class Status
 *
 * Integration worker for tomagento_integration_order_status jobs
 */
class Status extends Order
{
    /** @inheritDoc */
    protected $code = 'tomagento_integration_order_status';

    /** @var OrderConverter $orderConverter */
    protected $orderConverter;

    /** @var TrackFactory $trackFactory */
    protected $trackFactory;

    /** @var ShipmentRepository $shipmentRepository */
    protected $shipmentRepository;

    /** @var ShipmentNotifier $shipmentNotifier */
    protected $shipmentNotifier;

    /** @var string $hydrationWorker */
    protected $hydrationWorker = 'tomagento_hydration_order';

    /**
     * Status constructor
     *
     * @param Logger                $logger
     * @param JobRepository         $jobRepository
     * @param Config                $config
     * @param SyncedOrderRepository $syncedOrderRepository
     * @param OrderRepository       $orderRepository
     * @param OrderConverter        $orderConverter
     * @param TrackFactory          $trackFactory
     * @param ShipmentRepository    $shipmentRepository
     * @param ShipmentNotifier      $shipmentNotifier
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        SyncedOrderRepository $syncedOrderRepository,
        OrderRepository $orderRepository,
        OrderConverter $orderConverter,
        TrackFactory $trackFactory,
        ShipmentRepository $shipmentRepository,
        ShipmentNotifier $shipmentNotifier
    ) {
        parent::__construct($logger, $jobRepository, $config, $syncedOrderRepository, $orderRepository);

        $this->orderConverter = $orderConverter;
        $this->trackFactory = $trackFactory;
        $this->shipmentRepository = $shipmentRepository;
        $this->shipmentNotifier = $shipmentNotifier;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->notice($exception->getMessage());

            return;
        }

        foreach ($this->getJobs()->getItems() as $job) {
            if (!$job->isInError()) {
                try {
                    $this->processJob($job);
                } catch (JobException $exception) {
                    $this->invalidateJob($job, $exception);
                } catch (NoSuchEntityException | CouldNotShipException $exception) {
                    $this->skipJob($job, $exception);
                }

                $this->saveJob($job);
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @throws NoSuchEntityException
     * @throws CouldNotShipException
     */
    protected function processJob($job)
    {
        /** @var OrderEntity $entity */
        $entity = $job->getEntity();

        $order = $this->getMagentoOrder($entity->getFastmagId());

        if ($order->canShip() && $this->isOrderSent($entity->getStatus())) {
            $shipment = $this->createShipment($order);

            $trackData = $this->trackFactory->create();
            $trackData->setCarrierCode($order->getShippingMethod())
                ->setTitle($order->getShippingDescription())
                ->setNumber($entity->getTrackingNumber());

            $shipment->addTrack($trackData);

            try {
                /** @throws CouldNotSaveException */
                $this->shipmentRepository->save($shipment);
            } catch (CouldNotSaveException $exception) {
                throw new JobException(__(
                    'Unable to save the shipment: %1',
                    ($exception->getPrevious() ? $exception->getPrevious()->getMessage() : $exception->getMessage())
                ));
            }

            $this->orderRepository->save($order);

            try {
                $this->shipmentNotifier->notify($shipment);

                $this->shipmentRepository->save($shipment);
            } catch (MailException $exception) {
                throw new JobException(
                    __('Unable to send notify the customer of the shipment: %1', $exception->getMessage())
                );
            }
        } else {
            throw new CouldNotShipException(__('Order can\'t be shipped'));
        }
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_ORDER_IMPORT_STATUS_UPDATES);
    }

    /**
     * Returns Magento order object from transaction ID
     *
     * @param int $transactionId
     *
     * @return OrderModel
     *
     * @throws NoSuchEntityException
     */
    protected function getMagentoOrder($transactionId)
    {
        $syncedOrder = $this->syncedOrderRepository->getLastSyncByTransactionId($transactionId);

        $magentoId = $syncedOrder->getOrderId();

        return $this->orderRepository->get($magentoId);
    }

    /**
     * Tells if the order is considered as sent in Fastmag
     *
     * @param int $status
     *
     * @return bool
     */
    protected function isOrderSent($status)
    {
        return $status >= 300 && $status < 400;
    }

    /**
     * Create shipment from Magento order
     *
     * @param OrderModel $order
     *
     * @return ShipmentModel
     *
     * @throws JobException
     */
    protected function createShipment($order)
    {
        $shipment = $this->orderConverter->toShipment($order);

        foreach ($order->getAllItems() as $orderItem) {
            if (!$orderItem->getQtyToShip() || $orderItem->getIsVirtual()) {
                continue;
            }

            try {
                $shipmentItem = $this->orderConverter->itemToShipmentItem($orderItem)
                    ->setQty($orderItem->getQtyToShip());
            } catch (LocalizedException $exception) {
                throw new JobException(__('Unable to add item to shipment: %1', $exception->getMessage()));
            }

            $shipment->addItem($shipmentItem);
        }

        try {
            $shipment->register();
        } catch (LocalizedException $exception) {
            throw new JobException(__('Unable to save shipment: %1', $exception->getMessage()));
        }

        $shipment->getOrder()->setIsInProcess(true);

        return $shipment;
    }
}
