<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-27
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;

use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Exception\ProductNotSyncableException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToMagento\Product as ProductEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Product\Variation as VariationEntity;
use Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;
use Monolog\Logger as Monolog;

/**
 * Class Inventory
 *
 * Integration worker for tomagento_integration_product_quantity jobs
 */
class Inventory extends Product
{
    /** @inheritDoc */
    protected $code = 'tomagento_integration_product_inventory';

    /** @var string $hydrationWorker */
    protected $hydrationWorker = 'tomagento_hydration_product_inventory';

    /**
     * Configure importer
     *
     * @return void
     */
    protected function configureImporter()
    {
        parent::configureImporter();

        $this->importer->setEntityCode('stock_sources');
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->error('[' . $this->getCode() . '] ' . $exception->getMessage());
            return;
        }

        foreach ($this->getJobs()->getItems() as $job) {
            if (!$job->isInError()) {
                try {
                    $this->processJob($job);
                } catch (JobException $exception) {
                    $this->invalidateJob($job, $exception);
                } catch (ProductNotSyncableException $exception) {
                    $this->skipJob($job, $exception);
                }
            } else {
                $this->log($job, 'Job previously in error, skip integration.', Monolog::NOTICE);
            }
        }

        try {
            $this->runImport();
        } catch (JobException $exception) {
            foreach ($this->getJobs()->getItems() as $job) {
                $this->invalidateJob($job, $exception);
            }
        }

        foreach ($this->getJobs()->getItems() as $job) {
            if (!$job->isInError()) {
                $this->saveJob($job);
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @throws ProductNotSyncableException
     */
    protected function processJob($job)
    {
        $this->currentJob = $job;
        $this->currentDefaultData = [];

        /** @var ProductEntity $entity */
        $entity = $job->getEntity();
        $this->currentEntity = $entity;

        if ($this->productMustBeSynced($entity)) {
            $this->generateDefaultData();
            $simpleProductsData = $this->generateSimpleProductsData();
        } else {
            throw new ProductNotSyncableException(__(
                'Product "%1" can\'t be synced, it must be inactive or not in stock: %2',
                $entity->getRef(),
                $this->jsonSerializer->serialize($entity->export())
            ));
        }

        $this->moveDataToImport($simpleProductsData);
    }

    /**
     * @inheritDoc
     */
    protected function generateDefaultData()
    {
        $this->currentDefaultData = [
            'source_code' => $this->config->getValue(Config::XML_PATH_INVENTORY_MSI_SOURCE)
        ];
    }

    /**
     * @inheritDoc
     */
    protected function generateSimpleProductData($variation)
    {
        $result = [];

        $simpleProductData = [
            'sku'      => $this->generateSimpleProductSku($variation),
            'status'   => $this->getSimpleProductStatus($variation),
            'quantity' => $this->getSimpleProductStockLevel($variation)
        ];

        $result['default'] = array_merge(
            $this->currentDefaultData,
            $simpleProductData
        );

        $this->log(
            $this->currentJob,
            'Data to import for simple product "' . $simpleProductData['sku'] . '": '
            . $this->jsonSerializer->serialize($result)
        );

        return $result;
    }

    /**
     * @param VariationEntity $variation
     *
     * @return bool
     */
    protected function getSimpleProductStatus($variation)
    {
        return ($variation->getInStock() && $variation->getStockLevel() > 0);
    }

    /**
     * Check if there's no error on current data, then move them to the $dataToImport variable
     *
     * @param array $simpleProductsData
     *
     * @return void
     */
    protected function moveDataToImport($simpleProductsData)
    {
        if (count($simpleProductsData) > 0) {
            foreach ($simpleProductsData as $row) {
                $this->dataToImport[] = $this->exportDataToImport($row['default']);

                if (array_key_exists('store', $row)) {
                    foreach ($row['store'] as $storeRow) {
                        $this->dataToImport[] = $this->exportDataToImport($storeRow);
                    }
                }
            }
        }
    }
}
