<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-27
 ******************************************************************************/

namespace Fastmag\Sync\Process;

use Fastmag\Sync\Exception\NoJobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Jobqueue as AbstractJob;
use Fastmag\Sync\Model\Jobqueue\StandardRepository as JobRepository;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\Collection as JobCollection;
use Fastmag\Sync\Process\Worker\Hydration;
use Fastmag\Sync\Process\Worker\Integration;
use Fastmag\Sync\Process\Worker\Standard;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Manager
 *
 * Parent class of the process manager (remote sync, to magento and to fastmag)
 */
abstract class Manager
{
    /** @var Logger $logger */
    protected $logger;

    /** @var Config $config */
    protected $config;

    /** @var WorkerFactory $workerFactory */
    protected $workerFactory;

    /**
     * Job queue manager constructor.
     *
     * @param Logger        $logger
     * @param Config        $config
     * @param WorkerFactory $workerFactory
     */
    public function __construct(
        Logger $logger,
        Config $config,
        WorkerFactory $workerFactory
    ) {
        $this->logger = $logger;
        $this->config = $config;
        $this->workerFactory = $workerFactory;
    }

    /**
     * Get job repository
     *
     * @return JobRepository|null
     */
    abstract protected function getJobRepository();

    /**
     * Get current jobs collection
     *
     * @return JobCollection
     */
    abstract protected function getCurrentJobsCollection();

    /**
     * Check required config fields to run the synchronization
     *
     * @return bool
     *
     * @throws ProcessException
     */
    protected function checkPrerequisites()
    {
        $result = true;

        if (!$this->isSyncEnabled()) {
            throw new ProcessException(__('Synchronization is currently disabled. Remote sync can not be done.'));
        }

        if ($this->getSyncLimit() <= 0) {
            throw new ProcessException(__('Process queue limit size not set in config.'));
        }

        return $result;
    }

    /**
     * Check if global synchronization is enabled
     *
     * @return bool
     */
    protected function isSyncEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_JOBQUEUE_SYNC_ENABLED);
    }

    /**
     * Get sync limit
     *
     * @return int
     */
    protected function getSyncLimit()
    {
        return (int)$this->config->getValue(Config::XML_PATH_JOBQUEUE_ADVANCED_JOBS_LIMIT);
    }

    /**
     * Process jobs collection
     *
     * @param string        $jobCode
     * @param JobCollection $jobsCollection
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function processJobs($jobCode, $jobsCollection)
    {
        if ($jobsCollection->count() > 0) {
            $this->currentJobsCollection = $jobsCollection;

            $worker = $this->getJobsWorker($jobCode);

            if ($worker->isEnabled()) {
                $this->logger->debug(__('[%1] Beginning hydration', $worker->getHydrationWorker())->render());

                try {
                    $this->hydrateJobs($worker);
                    $this->logger->debug(
                        __('[%1] Hydration done, beginning integration.', $worker->getCode())->render()
                    );

                    $this->runWorker($worker);
                    $this->logger->debug(__('[%1] Integration done', $worker->getCode())->render());
                } catch (NoJobException $exception) {
                    $this->logger->debug(
                        __('[%1] %2', $worker->getCode(), $exception->getMessage())->render()
                    );
                }

                $this->validateJobs($worker);
            } else {
                $this->logger->debug(
                    __('[%1] Workers for jobs "%2" are not enabled.', $worker->getCode(), $jobCode)->render()
                );
            }
        }
    }

    /**
     * Return worker instance for the current job code
     *
     * @param string $jobCode
     *
     * @return Integration
     */
    protected function getJobsWorker($jobCode)
    {
        /** @var Integration $result */
        $result = $this->workerFactory->create($jobCode);

        return $result;
    }

    /**
     * Run worker and subordinate workers if the worker has any
     *
     * @param Standard $worker
     *
     * @return void
     *
     * @throws NoJobException
     * @throws ProcessException
     */
    protected function runWorker($worker)
    {
        if ($worker->hasSubordinateWorkersBefore()) {
            foreach ($worker->getSubordinateWorkersBefore() as $subWorkerCode) {
                /** @var Standard $subWorker */
                $subWorker = $this->workerFactory->create($subWorkerCode);

                $this->logger->debug(
                    __('[%1] Running subordinate worker "%2".', $worker->getCode(), $subWorkerCode)->render()
                );

                $this->runWorker($subWorker);
            }
        }

        $this->setJobsRunning();

        $worker->setJobs($this->getCurrentJobsCollection())->run();
        $this->logger->debug(
            __('[%1] Worker finished.', $worker->getCode())->render()
        );

        $this->removeJobsFromCurrentCollection();

        if ($worker->hasSubordinateWorkersAfter()) {
            foreach ($worker->getSubordinateWorkersAfter() as $subWorkerCode) {
                /** @var Standard $subWorker */
                $subWorker = $this->workerFactory->create($subWorkerCode);

                $this->logger->debug(
                    __('[%1] Running subordinate worker "%2".', $worker->getCode(), $subWorkerCode)->render()
                );

                $this->runWorker($subWorker);
            }
        }
    }

    /**
     * Run hydration workers to hydrate current jobs and add entities created to the jobs collection
     *
     * @param Integration $worker
     *
     * @return void
     *
     * @throws ProcessException
     * @throws NoJobException
     */
    protected function hydrateJobs($worker)
    {
        if ($this->getCurrentJobsCollection()->count() > 0) {
            $hydrationJobCode = $worker->getHydrationWorker();

            if ($hydrationJobCode !== null) {
                /** @var Hydration $hydrationWorker */
                $hydrationWorker = $this->workerFactory->create($hydrationJobCode);

                $this->runWorker($hydrationWorker);
            }
        }
    }

    /**
     * Set jobs collection in status running
     *
     * @return void
     */
    protected function setJobsRunning()
    {
        if ($this->getCurrentJobsCollection()->count() > 0) {
            foreach ($this->getCurrentJobsCollection()->getItems() as $job) {
                if (!in_array($job->getStatus(), [AbstractJob::STATUS_SKIPPED, AbstractJob::STATUS_ERROR], true)) {
                    $job->setStatus(AbstractJob::STATUS_RUNNING);

                    $this->getJobRepository()->save($job);
                }
            }
        }
    }

    /**
     * Remove skipped and on error jobs from collection
     *
     * @return void
     *
     * @throws NoJobException
     */
    protected function removeJobsFromCurrentCollection()
    {
        $skippedJobs = $this->getCurrentJobsCollection()->getItemsByColumnValue('status', AbstractJob::STATUS_SKIPPED);
        $errorJobs = $this->getCurrentJobsCollection()->getItemsByColumnValue('status', AbstractJob::STATUS_ERROR);

        $jobsToRemove = array_merge($skippedJobs, $errorJobs);

        /** @var AbstractJob $job */
        foreach ($jobsToRemove as $job) {
            $this->getCurrentJobsCollection()->removeItemByKey($job->getId());
        }

        if ($this->getCurrentJobsCollection()->count() === 0) {
            throw new NoJobException(__('No more jobs to process after removing skipped and on error jobs'));
        }
    }

    /**
     * Validate and save jobs
     *
     * @param Integration   $worker
     *
     * @return void
     */
    protected function validateJobs($worker)
    {
        $workerCode = $worker->getCode();

        if ($this->getCurrentJobsCollection()->count() > 0) {
            foreach ($this->getCurrentJobsCollection()->getItems() as $job) {
                if (!$job->isInError()) {
                    $this->getJobRepository()->validate($job);

                    try {
                        $this->getJobRepository()->save($job);
                    } catch (CouldNotSaveException $exception) {
                        $this->logger->critical(
                            '[' . $workerCode . '][Job #' . $job->getId() . '][Entity ' . $job->getContentId() . '] '
                            . '[ERROR] Unable to save job.'
                        );
                    }

                    $this->logger->debug(
                        '[' . $workerCode . '][Job #' . $job->getId() . '][Entity ' . $job->getContentId() . '] '
                        . 'Job validated'
                    );
                }
            }
        }
    }
}
