<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-10-19
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\Rule\TaxclassfamilyRepositoryInterface as TaxclassFamilyRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Importer;
use Fastmag\Sync\Process\Entity\ToMagento\Product as ProductEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Product\Variation as VariationEntity;
use Fastmag\Sync\Process\Worker;
use Fastmag\Sync\Process\Worker\ToMagento\Integration as IntegrationTrait;
use Magento\Eav\Api\AttributeSetRepositoryInterface as AttributeSetRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\ImportExport\Model\Import;
use Magento\ImportExport\Model\Import\ErrorProcessing\ProcessingErrorAggregatorInterface;
use Magento\Tax\Api\TaxClassRepositoryInterface as TaxClassRepository;
use Magento\Tax\Model\Config as TaxConfig;

/**
 * Class Product
 *
 * Abstract class for Product related Integration workers
 */
abstract class Product extends Worker
{
    use IntegrationTrait;

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /** @var Importer $importer */
    protected $importer;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var TaxClassRepository $taxClassRepository */
    protected $taxClassRepository;

    /** @var TaxclassFamilyRepository $taxclassFamilyRuleRepository */
    protected $taxclassFamilyRuleRepository;

    /** @var AttributeSetRepository $attributeSetRepository */
    protected $attributeSetRepository;

    /** @var Job $currentJob */
    protected $currentJob;

    /** @var ProductEntity $currentEntity */
    protected $currentEntity;

    /** @var array $dataToImport */
    protected $dataToImport = [];

    /** @var array $currentDefaultData */
    protected $currentDefaultData = [];

    /**
     * Save constructor
     *
     * @param Logger                   $logger
     * @param JobRepository            $jobRepository
     * @param Config                   $config
     * @param Json                     $jsonSerializer
     * @param Importer                 $importer
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param TaxClassRepository       $taxClassRepository
     * @param TaxclassFamilyRepository $taxclassFamilyRuleRepository
     * @param AttributeSetRepository   $attributeSetRepository
     */
    public function __construct(
        Logger $logger,
        JobRepository $jobRepository,
        Config $config,
        Json $jsonSerializer,
        Importer $importer,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        TaxClassRepository $taxClassRepository,
        TaxclassFamilyRepository $taxclassFamilyRuleRepository,
        AttributeSetRepository $attributeSetRepository
    ) {
        parent::__construct($logger);

        $this->jobRepository = $jobRepository;
        $this->config = $config;
        $this->jsonSerializer = $jsonSerializer;
        $this->importer = $importer;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->taxClassRepository = $taxClassRepository;
        $this->taxclassFamilyRuleRepository = $taxclassFamilyRuleRepository;
        $this->attributeSetRepository = $attributeSetRepository;

        $this->configureImporter();
    }

    /**
     * Configure importer
     *
     * @return void
     */
    protected function configureImporter()
    {
        $this->importer->setBehavior(Import::BEHAVIOR_APPEND)
            ->setEntityCode('catalog_product')
            ->setValidationStrategy(ProcessingErrorAggregatorInterface::VALIDATION_STRATEGY_SKIP_ERRORS)
            ->setAllowedErrorCount(100)
            ->setIgnoreDuplicates(false)
            ->setCategoryPathSeparator('|');
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ENABLE);
    }

    /**
     * Get job's entity ID
     *
     * @param Job $job
     *
     * @return string
     */
    protected function getJobEntityId($job)
    {
        /** @var ProductEntity $entity */
        $entity = $job->getEntity();

        return $entity->getRef();
    }

    /**
     * Check if products fill all the condition to be synced
     *
     * @param ProductEntity $entity
     *
     * @return bool
     */
    protected function productMustBeSynced($entity)
    {
        $result = ($entity->getActive() && $entity->getVisibleWeb())
            || !$this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ACTIVE_VISIBLEWEB);

        $onlyDefinedStocks = $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_DEFINED_STOCKS);
        if ($onlyDefinedStocks && $entity->getStockLevel() === 0) {
            $result = false;
        }

        if ($entity->getMagentoId()) {
            $result = true;
        }

        return $result;
    }

    /**
     * Generate default data for the entity
     *
     * @return void
     *
     * @throws JobException
     */
    abstract protected function generateDefaultData();

    /**
     * Returns the default attribute set code for product creation
     *
     * @return string
     *
     * @throws JobException
     */
    protected function getDefaultAttributeSetCode()
    {
        $defaultAttributeSetId = $this->config->getValue(Config::XML_PATH_PRODUCT_IMPORT_ATTRIBUTE_SET_ID);

        try {
            $defaultAttributeSet = $this->attributeSetRepository->get($defaultAttributeSetId);

            return $defaultAttributeSet->getAttributeSetName();
        } catch (NoSuchEntityException $exception) {
            throw new JobException(__(
                'Attribute set with ID #%1 not found in Magento. Please review you config.',
                $defaultAttributeSetId
            ));
        }
    }

    /**
     * Generate data to integrate for simple products
     *
     * @return array
     *
     * @throws JobException
     */
    protected function generateSimpleProductsData()
    {
        $result = [];

        $variations = $this->currentEntity->getVariationsFlat();

        foreach ($variations as $variation) {
            $simpleProductData = $this->generateSimpleProductData($variation);

            $result[$simpleProductData['default']['sku']] = $simpleProductData;
        }

        return $result;
    }
    /**
     * Generate data to integrate for simple product
     *
     * @param VariationEntity $variation
     *
     * @return array
     *
     * @throws JobException
     */
    abstract protected function generateSimpleProductData($variation);

    /**
     * Generate simple product name
     *
     * @param VariationEntity $variation
     * @param int|null        $storeId
     *
     * @return string
     */
    protected function generateSimpleProductName($variation, $storeId = null)
    {
        $result = $this->generateConfigurableProductName();

        if ($storeId !== null) {
            $parentI18n = $this->currentEntity->getI18n($storeId);

            if ($parentI18n !== null) {
                $result = $this->generateConfigurableProductName($parentI18n);
            }
        }

        if ($result !== '') {
            $result .= ' ' . $variation->getColor() . ' ' . $variation->getSize();
        }

        return trim($result);
    }

    /**
     * Generate configurable product name
     *
     * @param ProductEntity $entity
     *
     * @return string
     */
    protected function generateConfigurableProductName($entity = null)
    {
        if ($entity === null) {
            $entity = $this->currentEntity;
        }

        return trim($entity->getDesignation() . ' ' . $entity->getDesignationBis());
    }

    /**
     * Generate simple product SKU
     *
     * @param VariationEntity $variation
     *
     * @return string
     */
    protected function generateSimpleProductSku($variation)
    {
        $result = $this->currentEntity->getRef() . '-' . $variation->getColor() . '-' . $variation->getSize();

        if ($variation->getColor() === null) {
            $result = $this->currentEntity->getRef() . '-' . $variation->getSize();
        }

        return $result;
    }

    /**
     * Get price for a simple product
     *
     * @param VariationEntity $variation
     *
     * @return float
     */
    protected function getSimpleProductPrice($variation)
    {
        $result = $variation->getIndicativePrice();

        $standardPrice = (float)$variation->getStandardPrice();
        $combinationPrice = (float)$variation->getCombinationPrice();
        $ratePrice = (float)$variation->getRatePrice();

        if ($ratePrice > 0) {
            $result = $ratePrice;
        } elseif ($standardPrice > 0) {
            $result = $standardPrice;
        } elseif ($combinationPrice > 0) {
            $result = $combinationPrice;
        }

        return $this->formatPrice($result);
    }

    /**
     * Format price, including or excluding tax
     *
     * @param float $price
     *
     * @return float
     */
    protected function formatPrice($price)
    {
        $result = $price;

        if (!$this->config->isSetFlag(TaxConfig::CONFIG_XML_PATH_PRICE_INCLUDES_TAX)) {
            $vatRate = 20.0;

            if ($this->currentEntity->getShopVatRate()) {
                $vatRate = $this->currentEntity->getShopVatRate();
            }
            if ($this->currentEntity->getVatRate()) {
                $vatRate = $this->currentEntity->getVatRate();
            }

            $result = round($price / (1 + ($vatRate / 100)), 4);
        }

        return $result;
    }

    /**
     * Get global stock level, indicated by the stock level and the token is_in_stock for each store
     *
     * @param VariationEntity $variation
     *
     * @return int
     */
    protected function getSimpleProductStockLevel($variation)
    {
        $stockLevel = ($variation->getInStock() ? $variation->getStockLevel() : 0);

        $i18ns = $variation->getI18ns();
        if ($i18ns !== null && count($i18ns) > 0) {
            foreach ($i18ns as $i18n) {
                $stockLevel += $this->getSimpleProductStockLevel($i18n);
            }
        }

        return $stockLevel;
    }

    /**
     * Add required attributes values in imported data to bypass potential exceptions during import
     *
     * @param VariationEntity $variation
     *
     * @return array
     */
    protected function addRequiredAttributes($variation = null)
    {
        return [];
    }

    /**
     * Export data row in the format expected by the importer
     *
     * @return array
     */
    protected function exportDataToImport($row)
    {
        return $row;
    }

    /**
     * Run product importer
     *
     * @throws JobException
     */
    protected function runImport()
    {
        $this->logger->debug(__(
            '[%1] Data to import: %2',
            $this->getCode(),
            $this->jsonSerializer->serialize($this->dataToImport)
        )->render());

        try {
            if (count($this->dataToImport) > 0) {
                $this->importer->processImport($this->dataToImport);

                $this->logger->warning(__(
                    '[%1] Import results: %2',
                    $this->getCode(),
                    $this->importer->getLogTrace()
                )->render());
            }
        } catch (Exception $exception) {
            $message = $exception->getMessage() . "\n" . 'Import errors: ' . $this->importer->getErrorMessages();

            throw new JobException(__($message));
        }
    }
}
