<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-09-30
 ******************************************************************************/

namespace Fastmag\Sync\Process\Manager;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Jobqueue as AbstractJob;
use Fastmag\Sync\Model\Jobqueue\ToFastmagRepository as JobRepository;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToFastmag\Collection as JobCollection;
use Fastmag\Sync\Process\Manager;
use Fastmag\Sync\Process\Worker\ToFastmag\Clean;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration;
use Fastmag\Sync\Process\WorkerFactory;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class ToFastmag
 *
 * Workers manager for synchronization from Magento to Fastmag
 */
class ToFastmag extends Manager
{
    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var SortOrderBuilder $sortOrderBuilder */
    protected $sortOrderBuilder;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /**
     * ToFastmag constructor
     *
     * @param Logger                $logger
     * @param Config                $config
     * @param WorkerFactory         $workerFactory
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param SortOrderBuilder      $sortOrderBuilder
     * @param JobRepository         $jobRepository
     */
    public function __construct(
        Logger $logger,
        Config $config,
        WorkerFactory $workerFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SortOrderBuilder $sortOrderBuilder,
        JobRepository $jobRepository
    ) {
        parent::__construct($logger, $config, $workerFactory);

        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sortOrderBuilder = $sortOrderBuilder;
        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     *
     * @return JobRepository
     */
    protected function getJobRepository()
    {
        return $this->jobRepository;
    }

    /**
     * @inheritDoc
     *
     * @return JobCollection
     */
    protected function getCurrentJobsCollection()
    {
        return $this->currentJobsCollection;
    }

    /**
     * Main method
     *
     * @param string|null $jobCode
     * @param string|null $jobId
     *
     * @return void
     *
     * @throws Exception
     */
    public function run($jobCode = null, $jobId = null)
    {
        try {
            $checkOk = $this->checkPrerequisites($jobCode, $jobId);

            if ($checkOk) {
                $this->logger->debug('To Fastmag Sync - Begin process');

                $jobsCollectionByCode = $this->getJobsCollections($jobCode, $jobId);

                if (count($jobsCollectionByCode) > 0) {
                    foreach ($jobsCollectionByCode as $collectionJobCode => $jobsCollection) {
                        $this->processJobs($collectionJobCode, $jobsCollection);
                    }
                }

                /** @var Clean cleanWorker */
                $cleanWorker = $this->workerFactory->create('tofastmag_clean');
                $cleanWorker->run();
            }
        } catch (ProcessException $exception) {
            $this->logger->critical($exception->getMessage());

            throw $exception;
        }
    }

    /**
     * Get array of jobs collection to process
     *
     * @param string|null $jobCode
     * @param string|null $jobId
     *
     * @return array
     */
    protected function getJobsCollections($jobCode, $jobId)
    {
        $sortOrderJobId = $this->sortOrderBuilder->setField(Job::JOB_ID)
            ->setAscendingDirection()
            ->create();

        $this->searchCriteriaBuilder->addFilter(Job::STATUS, AbstractJob::STATUS_PENDING)
            ->addSortOrder($sortOrderJobId)
            ->setPageSize($this->getSyncLimit());

        if ($jobId) {
            $this->searchCriteriaBuilder->addFilter(Job::JOB_ID, $jobId);
        } elseif ($jobCode) {
            $this->searchCriteriaBuilder->addFilter(Job::JOB_CODE, $jobCode);
        }

        $searchCriteria = $this->searchCriteriaBuilder->create();

        return $this->getJobRepository()->getListByCode($searchCriteria);
    }

    /**
     * Check required config fields to run the synchronization
     *
     * @param string|null $jobCode
     * @param string|null $jobId
     *
     * @return bool
     *
     * @throws ProcessException
     */
    protected function checkPrerequisites($jobCode = null, $jobId = null)
    {
        $result = parent::checkPrerequisites();

        if ($result) {
            if ($jobId !== null) {
                try {
                    $job = $this->getJobRepository()->getById($jobId);
                    $jobCode = $job->getJobCode();
                } catch (Exception $exception) {
                    throw new ProcessException(__('Job ID #%1 does not exist in queue.', $jobId));
                }
            }

            if ($jobCode !== null) {
                /** @var Integration $worker */
                $worker = $this->workerFactory->create($jobCode);

                if (!$worker->isEnabled()) {
                    throw new ProcessException(__('Jobs "%1" are not enabled.', $jobCode));
                }
            }
        }

        return $result;
    }

    /**
     * Process jobs collection
     *
     * @param string        $jobCode
     * @param JobCollection $jobsCollection
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function processJobs($jobCode, $jobsCollection)
    {
        if ($jobsCollection->count() > 0) {
            $this->resetJobsError($jobsCollection);

            parent::processJobs($jobCode, $jobsCollection);
        }
    }

    /**
     * Reset jobs collection message and trace data
     *
     * @param JobCollection $jobsCollection
     *
     * @return void
     */
    protected function resetJobsError($jobsCollection)
    {
        foreach ($jobsCollection->getItems() as $key => $job) {
            $job->setMessage(null)
                ->setTrace(null);

            try {
                $this->getJobRepository()->save($job);
            } catch (CouldNotSaveException $exception) {
                $jobsCollection->removeItemByKey($key);
            }
        }
    }
}
